/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xdecode.c
 *       functions to control configuration space and target decoders
 * -----------------------------------------------------------------*/

#include <string.h>

#include <xpciapi.h>
#include <xloader.h>
#include <xaddrmap.h>

#include <xstrutil.h>

/* Pointer to target database */
#define DB_TARGET (bx_handlearray[handle].db->Exerciser.CompleterTarget)

/* Pointer to Config Space */
#define DB_CONFIG (DB_TARGET.ConfigSpace)

#define RO_BIT 0           /* Read-only Bit  for config-mask */
#define RW_BIT 1           /* Read-write Bit for config-mask */

/* Values for Bit 0 of a Bar decoder */
#define IO_DECODER  1
#define MEM_DECODER 0

/* Default masks and values for IO and memory decoders, when changing the location */
#define DEFAULT_MEM_MASK_LO 0xffff0000 /* 64 KB */
#define DEFAULT_MEM_MASK_HI 0xffffffff 
#define DEFAULT_MEM_VAL_LO  0xc; /* Set prefetch bit and type to 64 Bit mem space */
#define DEFAULT_MEM_VAL_HI  0x0; 

#define DEFAULT_IO_MASK  0xffff0000 /* 64 KB */
#define DEFAULT_IO_VAL   0x1; /* type to 32 Bit IO space */

static struct 
{
  bx_int32 Control,ResBase,ResSize;
} 
DecAddr[BX_DECODER_SIZE]=
{
  /* BAR0-5 */
  {BX_REG_STD0_CTRL_REG,BX_REG_STD0_RES_BASE_REG,BX_REG_STD0_RES_SIZE_REG},
  {BX_REG_STD1_CTRL_REG,BX_REG_STD1_RES_BASE_REG,BX_REG_STD1_RES_SIZE_REG},
  {BX_REG_STD2_CTRL_REG,BX_REG_STD2_RES_BASE_REG,BX_REG_STD2_RES_SIZE_REG},
  {BX_REG_STD3_CTRL_REG,BX_REG_STD3_RES_BASE_REG,BX_REG_STD3_RES_SIZE_REG},
  {BX_REG_STD4_CTRL_REG,BX_REG_STD4_RES_BASE_REG,BX_REG_STD4_RES_SIZE_REG},
  {BX_REG_STD5_CTRL_REG,BX_REG_STD5_RES_BASE_REG,BX_REG_STD5_RES_SIZE_REG},

  /* EXPROM */
  {BX_REG_EXPROM_CTRL_REG,BX_REG_EXPROM_RES_BASE_REG,BX_REG_EXPROM_RES_SIZE_REG}

  /* Remark: Config- and Requester decoders have no separate registers.
     They are handled completely by the Decoder enable register */
};

/* static function prototypes */
static bx_errtype BestXTDecoderBarSet(bx_handletype,bx_dectype,bx_decproptype,bx_int32);
static bx_errtype BestXTDecoderExpSet(bx_handletype,bx_decproptype,bx_int32);
static bx_errtype BestXTDecoderCfgSet(bx_handletype,bx_decproptype,bx_int32);
static bx_errtype BestXTDecoderReqSet(bx_handletype,bx_decproptype,bx_int32);
static bx_errtype BestXTDecoderBarSizeSet(bx_handletype,bx_dectype,bx_int32);
static bx_errtype BestXTDecoderBarLocationSet(bx_handletype,bx_dectype,bx_int32);
static bx_errtype BestXTDecoderBarPrefetchSet(bx_handletype,bx_dectype,bx_int32);

static bx_errtype BestXTDecoderBaseAddrSet (
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32ptr pBaseLo,
  bx_int32ptr pBaseHi
  );

static bx_errtype BestXTDecoderBarGet(bx_handletype,bx_dectype,bx_decproptype,bx_int32 *);
static bx_errtype BestXTDecoderExpGet(bx_handletype,bx_decproptype,bx_int32 *);
static bx_errtype BestXTDecoderCfgGet(bx_handletype,bx_decproptype,bx_int32 *);
static bx_errtype BestXTDecoderReqGet(bx_handletype,bx_decproptype,bx_int32 *);
static bx_errtype BestXTDecoderBarSizeGet(bx_handletype,bx_dectype,bx_int32 *);
static bx_errtype BestXTDecoderBarLocationGet(bx_handletype,bx_dectype,bx_int32 *);
static bx_errtype BestXTDecoderBarPrefetchGet(bx_handletype,bx_dectype,bx_int32 *);

static bx_errtype BestXTDecoderBaseAddrGet (
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32ptr pBaseLo,
  bx_int32ptr pBaseHi
  );

static bx_errtype TDecoderSanityCheck(
  bx_handletype handle,
  bx_dectype bar,
  bx_int32 *LocationLow,
  bx_int32 *LocationHigh,
  bx_int32 *SizeLow,
  bx_int32 *SizeHigh,
  bx_int32 *Prefetch
);

static bx_errtype TDecoderReadCheck (bx_handletype handle,
                                     bx_bool doCheck,
                                     bx_bool * pResult);

static bx_errtype ConfigValueRead ( bx_handletype handle,
                                       bx_int32ptr pValue );
static bx_errtype ConfigExtRead ( bx_handletype handle,
                                  bx_int32ptr pValue );
static bx_errtype ConfigMaskRead ( bx_handletype handle,
                                           bx_int32ptr pMask );
static bx_errtype ConfigExtMaskRead ( bx_handletype handle,
                                      bx_int32ptr pMask );

static bx_errtype ConfigValueProg ( bx_handletype handle,
                                       bx_int32ptr pValue );
static bx_errtype ConfigExtProg ( bx_handletype handle,
                                  bx_int32ptr pValue );
static bx_errtype ConfigMaskProg ( bx_handletype handle,
                                           bx_int32ptr pMask );
static bx_errtype ConfigExtMaskProg ( bx_handletype handle,
                                      bx_int32ptr pMask );

/* For enhanced decoder error-messages */
static char errmsg[BX_LASTERR_STRING];

/* -----------------------------------------------------------------
 * Configuration Space Programming Functions
 * ----------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * BestXConfRegSet
 *
 * Purpose: Set a value in config space (on host DB only)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXConfRegSet(
    bx_handletype handle,
    bx_int32 offset, /* DW-aligned byte address */
    bx_int32 val
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegSet [confregset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM(2, (0x40<= offset && offset <= 0x5c) || offset >= BX_CTCFG_MEMDEPTHMAX*4);

    DB_CONFIG.value[offset/4]=val;  
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegGet
 *
 * Purpose: Get a value from config space (on host DB only)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXConfRegGet(
    bx_handletype handle,
    bx_int32 offset, /* DW-aligned byte address into config space */
    bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegGet [confregget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM(2, (0x40<= offset && offset <= 0x5c) || offset >=BX_CTCFG_MEMDEPTHMAX*4);
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    *val=DB_CONFIG.value[offset/4];  
  }
    
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegMaskSet
 *
 * Purpose: Set a mask in config space (on host DB only)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXConfRegMaskSet(
    bx_handletype handle,
    bx_int32 offset,
    bx_int32 value
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegMaskSet [confregmaskset]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM(2, (0x40<= offset && offset <= 0x5c) || offset >=BX_CTCFG_MEMDEPTHMAX*4);

    DB_CONFIG.mask[offset/4]=value;  
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegMaskGet
 *
 * Purpose: Get a mask from config space (on host DB only)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXConfRegMaskGet(
    bx_handletype handle,
    bx_int32 offset,
    bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegMaskGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM(2, (0x40<= offset && offset <= 0x5c) || offset >=BX_CTCFG_MEMDEPTHMAX*4);
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    *val=DB_CONFIG.mask[offset/4];  
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegWrite
 *
 * Purpose: write value to config space (directly), will also change DB
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT  BestXConfRegWrite(
  bx_handletype handle,
  bx_int32      offset,                 
  bx_int32      value                      
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegWrite [confregwrite]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    bx_int32 confReg;
    /* the following does all the range/capa checking as well! */
    BX_TRY(BestXConfRegSet(handle, offset, value));

    if (offset < 0x40UL)
    {
      confReg = BX_REG_DEV_VEN_ID_REG + offset;
    }
    else
    {
      confReg = BX_REG_PCIX_CMD_REG + (offset-0x60UL);
    }
    
    /* confregset has checked parameters, boundary etc, so just shoot now */
    BX_TRY(BestXDirectRegWrite(handle, confReg, 4UL, value));
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegRead
 *
 * Purpose: Read config DW from config space of card, leaves DB alone
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT  BestXConfRegRead(
  bx_handletype handle,
  bx_int32      offset,                  
  bx_int32      *pValue                    
)
{
  /* Reads value of a DW in config space */

  BX_DECLARE_FUNCNAME("BestXConfRegRead [confregread]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    bx_int32 confReg;
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI); 

    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM(2, (0x40<= offset && offset <= 0x5c) || offset >=BX_CTCFG_MEMDEPTHMAX*4);
    BX_TRY_FCT_PARAM_NULL_POINTER(pValue);

    if (offset < 0x40UL)
    {
      confReg = BX_REG_DEV_VEN_ID_REG + offset;
    }
    else
    {
      confReg = BX_REG_PCIX_CMD_REG + (offset-0x60UL);
    }
    
    /* get value from hardware */
    BX_TRY(BestXDirectRegRead(handle, confReg, 4UL, pValue));
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegMaskWrite
 *
 * Purpose:  Sets mask of a DW in config space , also sets DB!
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT  BestXConfRegMaskWrite(
  bx_handletype handle,
  bx_int32      offset,                
  bx_int32      value
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegMaskWrite [confregmaskwrite]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    bx_int32 confReg;
    /* set in DB first, does capa/range checking as well! */
    BX_TRY(BestXConfRegMaskSet(handle, offset, value));

    if (offset < 0x40UL)
    {
      confReg = BX_REG_DEV_VEN_ID_MASK_REG + offset;
    }
    else
    {
      confReg = BX_REG_PCIX_CMD_MASK_REG + (offset-0x60UL);
    }
    
    /* maskset has done all error checking, just go now! */
    BX_TRY(BestXDirectRegWrite(handle, confReg, 4UL, value));
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXConfRegMaskRead(
 *
 * Purpose:  Reads mask of a DW in config space, leaves DB alone
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT  BestXConfRegMaskRead(
  bx_handletype handle,
  bx_int32      offset,
  bx_int32      *pValue                       
)
{
  BX_DECLARE_FUNCNAME("BestXConfRegMaskRead [confregmaskread]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    bx_int32 confReg;
    BX_TRY_LICENSE(BX_CAPABILITY_EXERCISER|BX_CAPABILITY_CAPI);

    BX_TRY_FAIL(offset & 0x03UL ? BX_E_NO_DWORD_BOUNDARY : BX_E_OK);
    BX_TRY_FCT_PARAM(2, (0x40<= offset && offset <= 0x5c) || offset >=BX_CTCFG_MEMDEPTHMAX*4);
    BX_TRY_FCT_PARAM_NULL_POINTER(pValue);

    if (offset < 0x40UL)
    {
      confReg = BX_REG_DEV_VEN_ID_MASK_REG + offset;
    }
    else
    {
      confReg = BX_REG_PCIX_CMD_MASK_REG + (offset-0x60UL);
    }
    
    BX_TRY(BestXDirectRegRead(handle, confReg, 4UL, pValue));
  }
  
  BX_ERRETURN(BX_TRY_RET);
} 

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXConfRegCheck (bx_handletype handle)
 *
 * Purpose      : check config registers agains database
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXTConfigCheck (bx_handletype handle,
                                     bx_bool * pResult)
{
  BX_DECLARE_FUNCNAME("BestXTConfigCheck []");
  
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN {
    bx_dbctcfgtype cfgCopy;

    BX_TRY_FCT_PARAM_NULL_POINTER (pResult);

    /* read standard config space */
    BX_TRY (ConfigValueRead ( handle, &(cfgCopy.value[0]) ));

    /* compare standard config space */
    *pResult = (memcmp (cfgCopy.value,
                        DB_CONFIG.value,
                        (BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX)*sizeof(bx_int32)) == 0) ? 1 : 0;

    if (*pResult)
    {
      BX_TRY (ConfigMaskRead ( handle, &(cfgCopy.mask[0]) ));
      /* compare standard config space mask */
      *pResult = (memcmp ( &(cfgCopy.mask[0x00]),
                           &(DB_CONFIG.mask[0x00]),
                            (BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX)*sizeof(bx_int32)) == 0) ? 1 : 0;    }
    
  }

  BX_ERRETURN(BX_TRY_RET);
}

/* ------------------------------------------------------------------------
 * PCI Target Programming Functions
 * ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
 * BestXTDecoderDefaultSet
 *
 * Purpose: Defaults all decoder properties for all decoders
 *                 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTDecoderDefaultSet(
  bx_handletype handle)
{  
  BX_DECLARE_FUNCNAME("BestXTDecoderDefaultSet []");
  
  BX_TRY_VARS_NO_PROG;
  bx_int32 val,mask,offset;

  /* The decoder properties BX_DECP_SIZE,BX_DECP_PREFETCH and
     BX_DECP_LOCATION are implicitely coded in config space
     (for BARs and expansion rom decoders; BX_DECP_SIZE is
     handled separately for config and requester decoder).

     But config space may contain complete nonsense and
     the TargetDecoder functions do not work on arbitray config spaces.
     So we first bring some registers into a defined default state
     by using the BestXConfReg[Mask]Set() functions.
     These implicitely sets the above mentioned properties for
     the six bar- and the expansion rom decoder.
  */

  BX_TRY_BEGIN
  {
    /* Set all six BAR registers to default in config space */
    for (offset=0x10;offset<=0x24;offset+=4)
    
    {
      /* This implicitely defaults the following decoder properties:
        BX_DECP_SIZE,
        BX_DECP_PREFETCH and
        BX_DECP_LOCATION
      */
      BX_TRY(BestXConfRegDefaultGet(handle,offset,&val));
      BX_TRY(BestXConfRegMaskDefaultGet(handle,offset,&mask));
      BX_TRY(BestXConfRegSet(handle,offset,val));
      BX_TRY(BestXConfRegMaskSet(handle,offset,mask));
    }

    /* Do same for Expansion ROM BAR */
    offset=0x30;
    BX_TRY(BestXConfRegDefaultGet(handle,offset,&val));
    BX_TRY(BestXConfRegMaskDefaultGet(handle,offset,&mask));
    BX_TRY(BestXConfRegSet(handle,offset,0));
    BX_TRY(BestXConfRegMaskSet(handle,offset,0));

    /* Config decoder: on */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_CONFIG,BX_DECP_SIZE,1));

    /* Requester decoder: on */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_RT,BX_DECP_SIZE,1));    

    /* Here we still have to handle the RESOURCE properties 
       for BARS and exprom decoders */

    /* BX_DECP_RESOURCE */       
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR0,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR1,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR2,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR3,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR4,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR5,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_EXPROM,BX_DECP_RESOURCE,BX_DECP_RESOURCE_MEM));

    /* BX_DECP_RESSIZE */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR0,BX_DECP_RESSIZE,20));   /* 1MB */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR1,BX_DECP_RESSIZE,20)); 
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR2,BX_DECP_RESSIZE,16));   /* 64KB */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR3,BX_DECP_RESSIZE,16)); 
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR4,BX_DECP_RESSIZE,16));   /* 64KB */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR5,BX_DECP_RESSIZE,16));   /* 64KB */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_EXPROM,BX_DECP_RESSIZE,16)); /* 64KB */

    /* BX_DECP_RESBASE */
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR0,BX_DECP_RESBASE,0));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR1,BX_DECP_RESBASE,0));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR2,BX_DECP_RESBASE,(1<<20)-(1<<16)));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR3,BX_DECP_RESBASE,(1<<20)-(1<<16)));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR4,BX_DECP_RESBASE,(1<<20)-(1<<16)));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_BAR5,BX_DECP_RESBASE,(1<<20)-(1<<16)));
    BX_TRY(BestXTDecoderSet(handle,BX_DEC_EXPROM,BX_DECP_RESBASE,0));
  }
  BX_ERRETURN(BX_TRY_RET);
}

bx_errtype EXPORT BestXTDecoderSet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_decproptype prop,
  bx_int32   val
)
{
  /* Sets a decoder property */

  BX_DECLARE_FUNCNAME("BestXTDecoderSet []");

  BX_TRY_VARS_NO_PROG;


  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_GEN, (bx_int32)decoder));
    
    switch (decoder)
    {
      case BX_DEC_BAR0:
      case BX_DEC_BAR1:
      case BX_DEC_BAR2:
      case BX_DEC_BAR3:
      case BX_DEC_BAR4:
      case BX_DEC_BAR5:
        BX_TRY(BestXTDecoderBarSet(handle,decoder,prop,val));
        break;
      case BX_DEC_EXPROM:
        BX_TRY(BestXTDecoderExpSet(handle,prop,val));
        break;
      case BX_DEC_CONFIG:
        BX_TRY(BestXTDecoderCfgSet(handle,prop,val));
        break;
      case BX_DEC_RT:
        BX_TRY(BestXTDecoderReqSet(handle,prop,val));
        break;
      default:
       assert (("Unknown decoder, range checking failed", 0));
       BX_TRY_FCT_PARAM_ERROR(decoder, "Unknown decoder");
        break;
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}


bx_errtype EXPORT BestXTDecoderGet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_decproptype prop,
  bx_int32 * val
)
{
  /* Get a decoder property */
  BX_DECLARE_FUNCNAME("BestXTDecoderGet [tdecget]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_GEN, (bx_int32)decoder));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    switch (decoder)
    {
      case BX_DEC_BAR0:
      case BX_DEC_BAR1:
      case BX_DEC_BAR2:
      case BX_DEC_BAR3:
      case BX_DEC_BAR4:
      case BX_DEC_BAR5:
        BX_TRY(BestXTDecoderBarGet(handle,decoder,prop,val));
        break;
      case BX_DEC_EXPROM:
        BX_TRY(BestXTDecoderExpGet(handle,prop,val));
        break;
      case BX_DEC_CONFIG:
        BX_TRY(BestXTDecoderCfgGet(handle,prop,val));
        break;
      case BX_DEC_RT:
        BX_TRY(BestXTDecoderReqGet(handle,prop,val));
        break;
      default:
        assert (("Unknown decoder, range checking failed", 0));
        BX_TRY_FCT_PARAM_ERROR(decoder, "Unknown decoder");
        break;
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/* ------------------------------------------------------------------------
 * static PCI Target Programming Functions
 * ------------------------------------------------------------------------ */

static bx_errtype BestXTDecoderBarSet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_decproptype prop,
  bx_int32 val)
{
  /* Sets a property for a BAR decoder */

  BX_DECLARE_FUNCNAME("BestXTDecoderBarSet []");

  BX_TRY_VARS_NO_PROG;

  /* Parameter decoder is a valid BAR-decoder here.
     This is granted by the caller.
  */
  assert(decoder<=BX_DEC_BAR5);
  
  BX_TRY_BEGIN 
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_STD,(bx_int32)prop,val));

    switch (prop)
    {
     case BX_DECP_SIZE:
      BX_TRY(BestXTDecoderBarSizeSet(handle,decoder,val));
      break;
     case BX_DECP_PREFETCH:
      BX_TRY(BestXTDecoderBarPrefetchSet(handle,decoder,val));
      break;
     case BX_DECP_LOCATION:
      BX_TRY(BestXTDecoderBarLocationSet(handle,decoder,val));
      break;
     case BX_DECP_BASEADDR:
      BX_TRY (BestXTDecoderBaseAddrSet (handle, decoder, &val, NULL));
      break;
     case BX_DECP_BASEADDR_HI:
      BX_TRY (BestXTDecoderBaseAddrSet (handle, decoder, NULL, &val));
      break;
     default:
      /* All decoder properties not stored in config space
         (e.g. RESSIZE,RESBASE,RESOURCE).
      */
      /* Validate array boundaries (checked by dyncaps before).*/
      assert(decoder<BX_DECODER_SIZE);
      assert(prop<BX_DECPROP_SIZE);
      DB_TARGET.DecProp[decoder][prop]=val; 
      break;
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderBarGet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_decproptype prop,
  bx_int32 *val)
{
  /* Gets a property for a BAR decoder */

  BX_DECLARE_FUNCNAME("BestXTDecoderBarGet []");

  BX_TRY_VARS_NO_PROG;

  /* decoder is a valid BAR-decoder here (granted by caller). */
  assert(decoder<=BX_DEC_BAR5);
  
  BX_TRY_BEGIN 
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_STD,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
  
    switch (prop)
    {
     case BX_DECP_SIZE:
      BX_TRY(BestXTDecoderBarSizeGet(handle,decoder,val));
      break;

     case BX_DECP_PREFETCH:
      BX_TRY(BestXTDecoderBarPrefetchGet(handle,decoder,val));
      break;

     case BX_DECP_LOCATION:
      BX_TRY(BestXTDecoderBarLocationGet(handle,decoder,val));
      break;

     case BX_DECP_BASEADDR:
      BX_TRY (BestXTDecoderBaseAddrGet (handle, decoder, val, NULL));
      break;

     case BX_DECP_BASEADDR_HI:
      BX_TRY (BestXTDecoderBaseAddrGet (handle, decoder, NULL, val));
      break;
     
     default:
      /* All decoder properties not stored in config space
         (e.g. RESSIZE,RESBASE).
      */
      /* Validate array boundaries (checked by dyncaps before).*/

      assert(decoder<BX_DECODER_SIZE);
      assert(prop<BX_DECPROP_SIZE);
      *val=DB_TARGET.DecProp[decoder][prop]; 
      break;
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderExpSet(
  bx_handletype handle,
  bx_decproptype prop,
  bx_int32 val)
{
  /* Sets a property for the expansion rom decoder */

  BX_DECLARE_FUNCNAME("BestXTDecoderExpSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_EXP,(bx_int32)prop,val));
  
    switch (prop)
    {
     case BX_DECP_SIZE:
      /* This property is stored directly in the Config-Space */

      DB_CONFIG.value[12]=0;
      DB_CONFIG.mask [12]=0;  /* all Read only, decoder off */
      if (val)
      {
        /* switch on Expansion ROM decoder */
        assert(val>=11 && val <=24 || val==0); /* checked by dyncaps */
        DB_CONFIG.mask [12]=~((1<<val)-1);
        /* Bit 0 also needs write-access for BIOS: */
        DB_CONFIG.mask [12] |= 1;
      }

      /* Write invalid value to property
         (we do not want redundant data): 
      */
      DB_TARGET.DecProp[BX_DEC_EXPROM][prop]=0xdeadbeaf; 
      break;

     case BX_DECP_BASEADDR:
      BX_TRY (BestXTDecoderBaseAddrSet (handle, BX_DEC_EXPROM, &val, NULL));
      break;
    
     default:
      /* All props not stored in config space go here */
      /* Assert array boundaries: */
      assert(BX_DEC_EXPROM<BX_DECODER_SIZE);
      assert(prop<BX_DECPROP_SIZE);
      DB_TARGET.DecProp[BX_DEC_EXPROM][prop]=val;
      break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderExpGet(
  bx_handletype handle,
  bx_decproptype prop,
  bx_int32 *val)
{
  /* Gets a property for the expansion rom decoder */

  bx_int32 i;
  BX_DECLARE_FUNCNAME("BestXTDecoderExpGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_EXP,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    switch (prop)
    {
      case BX_DECP_SIZE:
      {
        /* This property is stored directly in the Config-Space */

        if ((DB_CONFIG.mask[12] & 1)==0)
        {
          /* Mask Bit 0 is Read-only, i.e. decoder off */
          *val=0;
        }
        else
        {
          /* mask Bit 0 is writable, determine size from 
             mask bits 
           */
          for (i=1;i<=32;i++)
          {
            if (DB_CONFIG.mask[12] & 1<<i)
            {
              /* i-th Bit is RW */
              break;
            }
          }
         
          /* i points to first RW bit or to 32 */

          if (i<11)
          {
            /* Bits 1,..,10 are reserved RO (PCI spec) */
            BX_TRY_FAIL(BX_E_CONFIG_MASK_INVALID);
          }

          if (i==32)
          {
            /* all bits are RO, maybe return an error here,
               because bit 0 is RW ??
             */
            *val=0;
          }
          else
          {
            *val=i; /* valid, nonzero size > 1 */
          }

          /* Make sure that the significant bits are all RW */
          for (i=i+1;i<=31;i++)
          {
            if ((DB_CONFIG.mask[12] & 1<<i)==0)
            {
              /* i-th Bit is RO again */
              BX_TRY_FAIL(BX_E_CONFIG_MASK_INVALID);
            }
          }
        }
      }   /* case BX_DECP_SIZE */
      break;
     case BX_DECP_BASEADDR:
      BX_TRY (BestXTDecoderBaseAddrGet (handle, BX_DEC_EXPROM, val, NULL));
      break;

     default:
        /* All props not stored in config space go here */
        assert(BX_DEC_EXPROM<BX_DECODER_SIZE);
        assert(prop<BX_DECPROP_SIZE);
        *val=DB_TARGET.DecProp[BX_DEC_EXPROM][prop];
        break;
    }     /* switch */
  }       /* TRY */
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderCfgSet(
  bx_handletype handle,
  bx_decproptype prop,
  bx_int32 val)
{
  /* Sets a property for the config decoder */
  BX_DECLARE_FUNCNAME("BestXTDecoderCfgSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_CFG,(bx_int32)prop,val));
  }
  
  assert(BX_DEC_CONFIG<BX_DECODER_SIZE);
  assert(prop<BX_DECPROP_SIZE);

  DB_TARGET.DecProp[BX_DEC_CONFIG][prop]=val;

  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderCfgGet(
  bx_handletype handle,
  bx_decproptype prop,
  bx_int32 *val)
{
  /* Gets a property for the config decoder */

  BX_DECLARE_FUNCNAME("BestXTDecoderCfgGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_CFG,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
  }
  
  assert(BX_DEC_CONFIG<BX_DECODER_SIZE);
  assert(prop<BX_DECPROP_SIZE);

  *val=DB_TARGET.DecProp[BX_DEC_CONFIG][prop];
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderReqSet(
  bx_handletype handle,
  bx_decproptype prop,
  bx_int32 val)
{
  /* Sets a property for the requester decoder */

  BX_DECLARE_FUNCNAME("BestXTDecoderReqSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_REQ,(bx_int32)prop,val));
  }
  
  assert(BX_DEC_RT<BX_DECODER_SIZE);
  assert(prop<BX_DECPROP_SIZE);

  DB_TARGET.DecProp[BX_DEC_RT][prop]=val;

  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderReqGet(
  bx_handletype handle,
  bx_decproptype prop,
  bx_int32 *val)
{
  /* Gets a property for the requester decoder */

  BX_DECLARE_FUNCNAME("BestXTDecoderReqGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_REQ,(bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
  }
  
  assert(BX_DEC_RT<BX_DECODER_SIZE);
  assert(prop<BX_DECPROP_SIZE);

  *val=DB_TARGET.DecProp[BX_DEC_RT][prop];
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderBarSizeSet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 val)
{
  /* Sets property BX_DECP_SIZE for a BAR decoder */

  /* Return parameters from TDecoderSanityCheck() */
  bx_int32 LocLow,LocHigh;
  bx_int32 SizeLow,SizeHigh;

  BX_DECLARE_FUNCNAME("BestXTDecoderBarSizeSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_STD,(bx_int32)BX_DECP_SIZE,val));
    BX_TRY(TDecoderSanityCheck(handle,decoder,&LocLow,&LocHigh,&SizeLow,&SizeHigh,NULL));

    /* The following if-cascadation (sorry) makes sure that we catch every case. 
       To enhance readability the current "if-stack" is described by assertions.
    */
 
    if (LocLow==IO_DECODER)
    {
      assert(LocLow==IO_DECODER);

      /* Lower BAR is an IO decoder */
      
      if (decoder%2)
      {
        assert(LocLow==IO_DECODER);
        assert(decoder==1||decoder==3||decoder==5);
       
        /* odd BAR decoder */

        if (LocHigh==MEM_DECODER)
        {
          assert(LocLow==IO_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(LocHigh==MEM_DECODER);
          
          assert(SizeHigh==0); /* checked by TDecoderSanityCheck() */
           
          if (val)
          {
            /* User wants to change the size of the higher BAR,
               which is a zero-sized memory decoder.
               This is forbidden. User should first
               change this BAR to IO-location or create a memory
               decoder at the lower BAR.
            */

            BESTX_SPRINTF(errmsg,
                          "Cannot create a MEM decoder at an odd BAR (%lu) (File:%s,Line:%lu)",
                          decoder,__FILE__,__LINE__);

            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_DECODER);
          }
          else
          {
            /* Decoder remains switched off, so this is OK */
          }
        }
        else
        {
          assert(LocLow==IO_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(LocHigh==IO_DECODER);

          /* Higher BAR is IO decoder, too */

          /* User wants to change the size of the higher BAR,
             which is an IO decoder.
             Mask Bits 2-31 determine the size.
          */
           
          /* Check range of size */
          BX_TRY_FCT_PARAM(3,!(val!=1&&val<32));
          
          if (val!=SizeHigh)
          {
            /* Set the new size */
            DB_CONFIG.mask  [decoder+4]= (val?~((1<<val)-1):0); 
          
            /* We try not to preserve the previous value.
               Caution: Zero sized IO decoders are forbidden,
               so we set the mask to all zero (zero sized 
               MEM decoder), if size=0.
            */

            DB_CONFIG.value [decoder+4]=(val?1:0); 
          }
          else
          {
            assert(val==SizeHigh);
            /* BX_DECP_SIZE already at the wanted size.
               We dont do anything here in order to preserve
               the current Base-address 
               (DB_CONFIG.value[] is not changed).
            */
          }
        }
      }
      else
      {
        assert(LocLow==IO_DECODER);
        assert(decoder==0||decoder==2||decoder==4);
       
        /* User wants to change the size of the lower BAR,
           which is an IO decoder.
           Mask Bits 2-31 determine the size.
        */

        /* Check range of size */
        BX_TRY_FCT_PARAM(3,!(val!=1&&val<32));
        
        if (val!=SizeLow)
        {
          /* Set the new size */
          DB_CONFIG.mask  [decoder+4]= (val?~((1<<val)-1):0); 
        
          /* We try not to preserve the previous value.
             Caution: Zero sized IO decoders are forbidden,
             so we set the mask to all zero (zero sized 
             MEM decoder), if size=0.
          */

          DB_CONFIG.value [decoder+4]=(val?1:0); 
        }
        else
        {
          assert(val==SizeLow);
          /* BX_DECP_SIZE already at the wanted size.
             We dont do anything here in order to preserve
             the current Base-address
             (DB_CONFIG.value[] is not changed).
          */
        }
      }
    }
    else
    {
      assert(LocLow==MEM_DECODER);
      
      /* BAR pair is a memory decoder */

      if (decoder%2)
      {
        assert(LocLow==MEM_DECODER);
        assert(decoder==1||decoder==3||decoder==5);
     
        /* User wants to change the size of the higher BAR,
           which is not allowed, if lower BAR is a 
           non-zero sized MEM-decoder covering both BARs.
           User should change the size of 
           the lower BAR instead. 
        */

        if (SizeLow)
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(SizeLow);

          /* User wants to change the size of the higher BAR,
             which is not allowed, because lower BAR is a 
             non-zero sized MEM-decoder covering both BARs.
             User should change the size of 
             the lower BAR instead. 
          */
          BESTX_SPRINTF(errmsg,"Cannot set size of upper BAR (%lu) of a memory decoder (File:%s,Line:%lu)",decoder,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_DECODER);
        }
        else
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(SizeLow==0);
           
          /* We have a zero-sized memory decoder at the lower BAR.
             There maybe an IO decoder or a zero sized MEM decoder at the 
             higher BAR.
          */

          if (LocHigh==IO_DECODER)
          {
            assert(LocLow==MEM_DECODER);
            assert(decoder==1||decoder==3||decoder==5);
            assert(SizeLow==0);
            assert(LocHigh==IO_DECODER);
            
            /* Got a zero-sized mem decoder at the lower BAR 
               and an IO decoder at the higher BAR.
               User wants to change the size of the IO decoder.
            */

            /* Check range of size */
            BX_TRY_FCT_PARAM(3,!(val!=1&&val<32));
        
            if (val!=SizeHigh)
            {
              assert(LocLow==MEM_DECODER);
              assert(decoder==1||decoder==3||decoder==5);
              assert(SizeLow==0);
              assert(LocHigh==IO_DECODER);
              assert(val!=SizeHigh);

              /* Set the new size */
              DB_CONFIG.mask [decoder+4]= (val?~((1<<val)-1):0); 
        
              /* We try not to preserve the previous value.
                 Caution: Zero sized IO decoders are forbidden,
                 so we set the mask to all zero (zero sized 
                 MEM decoder), if size=0.
              */

              DB_CONFIG.value [decoder+4]=(val?1:0); 
            }
            else
            {
              assert(LocLow==MEM_DECODER);
              assert(decoder==1||decoder==3||decoder==5);
              assert(SizeLow==0);
              assert(LocHigh==IO_DECODER);
              assert(val==SizeHigh);

              /* BX_DECP_SIZE already at the wanted size.
                 We dont do anything here in order to preserve
                 the current base-address
                 (DB_CONFIG.value[] is not changed).
              */
            }
          }
          else
          {
            assert(LocLow==MEM_DECODER);
            assert(decoder==1||decoder==3||decoder==5);
            assert(SizeLow==0);
            assert(LocHigh==MEM_DECODER);
            
            /* Got a zero-sized mem-decoder at the lower BAR
               and a memory decoder at the higher BAR.
               User wants to change the size of the higher BAR.
            */
             
            assert(SizeHigh==0); /* checked by TDecoderSanityCheck() */

            if (val)
            {
              BESTX_SPRINTF(errmsg,
                            "Cannot set size of upper BAR (%lu) of a memory decoder (File:%s,Line:%lu)",
                            decoder,__FILE__,__LINE__);
              (void) BestXLastErrorParamStringSet(handle,errmsg);
              BX_ERRETURN(BX_E_DECODER);
            }
            else
            {
              /* size remains zero, so we allow this NOP operation */
            }
          }
        }
      }
      else
      {
        assert(LocLow==MEM_DECODER);
        assert(decoder==0||decoder==2||decoder==4);
       
        /* User wants to change the size of the lower BAR,
           which is a MEM-decoder.
           Mask Bits 4-63 determine the size.
        */

        if (LocHigh==IO_DECODER)
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==0||decoder==2||decoder==4);
          assert(LocHigh==IO_DECODER);

          /* There is a (non-negative) IO decoder at the higher BAR,
             which needs to be switched off, before the MEM decoder's
             size can be changed to a positive value */
          
          assert(SizeLow==0); /* checked by TDecoderSanityCheck() */
          assert(SizeHigh);   /* checked by TDecoderSanityCheck() */

          /* There is a switched on IO decoder at the higher BAR.
             User should first switch off that one */
          BESTX_SPRINTF(errmsg,
                        "Cannot set size of memory decoder at BAR %lu to a"
                        " non-negative value because there is an IO decoder"
                        " at BAR %lu, which needs to be switched off "
                        "first (File:%s,Line:%lu)",
                        decoder,decoder+1,__FILE__,__LINE__);

          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_DECODER);
        }
        else
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==0||decoder==2||decoder==4);
          assert(LocHigh==MEM_DECODER);

          /* higher BAR is also a MEM decoder */
        
          /* Check range of size */
          /* Minimum memory decoder size for PCI-X is 128 bytes */
          BX_TRY_FCT_PARAM(3,!(val==0||val>=7&&val<64));
      
          /* Set the new size */
          if (val!=SizeLow)
          {
            if (val==0)
            {
              /* switch mem-decoder off */
              DB_CONFIG.mask   [decoder+4]=0; 
              DB_CONFIG.value  [decoder+4]=0; 
              DB_CONFIG.mask   [decoder+1+4]=0; 
              DB_CONFIG.value  [decoder+1+4]=0; 
            }
            else 
            {
              if (val<32)
              {
                DB_CONFIG.mask   [decoder+4]=~((1<<val)-1); 
                DB_CONFIG.value  [decoder+4]=0xc; /* Set prefetch and 64 bit space */ 
                DB_CONFIG.mask   [decoder+1+4]=0xffffffff; 
                DB_CONFIG.value  [decoder+1+4]=0; 
              }
              else
              {
                DB_CONFIG.mask   [decoder+4]=0; 
                DB_CONFIG.value  [decoder+4]=0xc; /* Set prefetch and 64 bit space */ 
                DB_CONFIG.mask   [decoder+1+4]=~((1<<(val-32))-1); 
                DB_CONFIG.value  [decoder+1+4]=0; 
              }
            }
          }        
          else
          {
            assert(val==SizeLow);
            /* BX_DECP_SIZE already at the wanted size.
               We dont do anything here in order to preserve
               the current Base-address
               (DB_CONFIG.value[] is not changed).
            */
          }
        }
      }
    }
  }    /* TRY */
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderBarLocationSet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 val)
{
  /* Sets property BX_DECP_LOCATION for a BAR decoder */

  bx_int32 LocLow,LocHigh;
  bx_int32 SizeLow,SizeHigh;

  BX_DECLARE_FUNCNAME("BestXTDecoderBarLocationSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_STD,(bx_int32)BX_DECP_LOCATION,val));

    if (val==BX_DECP_LOCATION_OFF)
    {
      /* Switch decoder off */
      BX_TRY(BestXTDecoderBarSizeSet(handle,decoder,0));
      BX_ERRETURN(BX_E_OK);
   }

    BX_TRY(TDecoderSanityCheck(handle,decoder,&LocLow,&LocHigh,&SizeLow,&SizeHigh,NULL));

    /* The following if-cascadation (sorry) makes sure that we catch every case. 
       To enhance readability the current "if-stack" is described by assertions.
    */
 
    if (LocLow==IO_DECODER)
    {
      assert(LocLow==IO_DECODER);
      
      /* Lower BAR contains an IO decoder */
      
      if (decoder%2)
      {
        assert(LocLow==IO_DECODER);
        assert(decoder==1||decoder==3||decoder==5);
       
        if (val==BX_DECP_LOCATION_IO)
        {
          assert(LocLow==IO_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(val==BX_DECP_LOCATION_IO);

          /* Someone wants an IO decoder */

          if (LocHigh==IO_DECODER)
          {
            assert(LocLow==IO_DECODER);
            assert(decoder==1||decoder==3||decoder==5);
            assert(val==BX_DECP_LOCATION_IO);
            assert(LocHigh==IO_DECODER);
             
            /* Someone wants to create an IO decoder at the higher BAR.
               Because there is already an IO decoder at the higher BAR
               (and at the lower BAR),
               we dont have to change anything here.
               Especially, we also leave the size (SizeHigh) unchanged 
               (dont set it to zero).
            */
          }
          else
          {
            assert(LocLow==IO_DECODER);
            assert(decoder==1||decoder==3||decoder==5);
            assert(val==BX_DECP_LOCATION_IO);
            assert(LocHigh==MEM_DECODER);
            
            /* Someone wants to create an IO decoder at the higher BAR.
               Got an IO decoder at lower BAR and a zero-sized
               memory decoder at the higher BAR
            */
            
            assert(SizeHigh==0); /* Checked by TDecoderSanityCheck() */

            /* Create an IO decoder with default size at the higher BAR 
               Caution: IO decoder of size zero is not allowed by Spec !! */
            DB_CONFIG.value[decoder+4]=DEFAULT_IO_VAL;
            DB_CONFIG.mask [decoder+4]=DEFAULT_IO_MASK; 
          }
        }
        else
        {
          assert(LocLow==IO_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(val==BX_DECP_LOCATION_MEM);

          /* IO decoder at the lower BAR */

          if (LocHigh==IO_DECODER)
          {
            assert(LocLow==IO_DECODER);
            assert(decoder==1||decoder==3||decoder==5);
            assert(val==BX_DECP_LOCATION_MEM);
            assert(LocHigh==IO_DECODER);
             
            /* Got two IO-decoders.
               Someone wants to create a MEM decoder at the higher BAR.
               This is not allowed by PCI-X Spec (would need an even BAR).
            */
            BESTX_SPRINTF(errmsg,
                          "Cannot create a MEM decoder at an odd BAR (%lu) (File:%s,Line:%lu)",
                          decoder,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_DECODER);
          }
          else
          {
            assert(LocLow==IO_DECODER);
            assert(decoder==1||decoder==3||decoder==5);
            assert(val==BX_DECP_LOCATION_MEM);
            assert(LocHigh==MEM_DECODER);
            
            /* Got an IO decoder at lower BAR and a zero-sized
               MEM-decoder at higher BAR. 
               Someone wants to create MEM decoder at the higher BAR.
               We issue an error here, although we could 
               leave the zero-sized memory decoder unchanged
               and do a NOP here (issuing an error later,
               when user tries to change the size to a nonzero value).
            */
            
            assert(SizeHigh==0); /* Checked by TDecoderSanityCheck() */

            BESTX_SPRINTF(errmsg,
                          "Cannot create a memory decoder at an odd BAR (%lu). (File:%s,Line:%lu)",
                          decoder,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_DECODER);
          }
        }
      }
      else
      {
        assert(LocLow==IO_DECODER);
        assert(decoder==0||decoder==2||decoder==4);
       
        /* Lower BAR contains IO decoder */

        if (val==BX_DECP_LOCATION_IO)
        {
          assert(LocLow==IO_DECODER);
          assert(decoder==0||decoder==2||decoder==4);
          assert(val==BX_DECP_LOCATION_IO);

          /* Someone wants to create an IO decoder at the lower BAR.
             Because there is already an IO decoder at the lower BAR
             we dont have to change anything here.
             Especially, we also leave the size (SizeLow) unchanged 
             (dont set it to zero).
          */
        }
        else
        {
          assert(LocLow==IO_DECODER);
          assert(decoder==0||decoder==2||decoder==4);
          assert(val==BX_DECP_LOCATION_MEM);

          /* Got an IO decoder at lower BAR and an IO or zero-sized
             memory decoder at higher BAR.
             Someone wants to create a MEM decoder at the lower BAR
          */
          
          if (LocHigh==IO_DECODER)
          {
            assert(SizeHigh); /* checked by TDecoderSanityCheck() */
    
            /* This is for Target Decode Window:
               We would overwrite a hidden, switched on
               IO decoder at the higher BAR.
               Instead we issue an error here, requesting
               the user to first switch off the IO decoder at BAR decoder+1
               (the high BAR).
            */
            BESTX_SPRINTF(errmsg,"Before creating a memory decoder at BAR pair (%lu,%lu), you first have to switch off the IO decoder at BAR %lu (File:%s,Line:%lu)",decoder,decoder+1,decoder+1,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_DECODER);
          }
          else
          {          
            /* Create a MEM decoder with default size at the lower BAR */
            DB_CONFIG.value[decoder+4]=DEFAULT_MEM_VAL_LO; 
            DB_CONFIG.mask [decoder+4]=DEFAULT_MEM_MASK_LO;  
            /* Higher BAR */
            DB_CONFIG.value[decoder+1+4]=DEFAULT_MEM_VAL_HI;
            DB_CONFIG.mask [decoder+1+4]=DEFAULT_MEM_MASK_HI; 
          }
        }
      }          
    }
    else
    {
      assert(LocLow==MEM_DECODER);
      
      /* Got a memory decoder at the lower BAR */

      if (decoder%2)
      {
        assert(LocLow==MEM_DECODER);
        assert(decoder==1||decoder==3||decoder==5);
       
        if (val==BX_DECP_LOCATION_IO)
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(val==BX_DECP_LOCATION_IO);

          /* Got a 64 bit decoder at the lower bar (BAR decoder-1).
             Someone wants to create an IO decoder at the higher BAR
             (BAR decoder).
          */

          if (SizeLow)
          {
            /* We would overwrite a hidden, switched on
               MEM decoder at the lower BAR.
               Instead we issue an error here, requesting
               the user to first switch off the MEM decoder at BAR decoder-1
               (the low BAR).
            */
            BESTX_SPRINTF(errmsg,"Before creating an IO decoder at BAR %lu, you first have to switch off the memory decoder at BAR pair (%lu,%lu) (File:%s,Line:%lu)",decoder,decoder-1,decoder,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_DECODER);
          }
          else
          {
            /* Create default IO decoder at higher BAR and zero-sized decoder
               at lower BAR */
            /* high BAR */
            DB_CONFIG.value[decoder+4]=DEFAULT_IO_VAL;   
            DB_CONFIG.mask [decoder+4]=DEFAULT_IO_MASK;
            /* Low BAR */
            DB_CONFIG.value[decoder-1+4]=0; 
            DB_CONFIG.mask [decoder-1+4]=0; 
          }
        }
        else
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==1||decoder==3||decoder==5);
          assert(val==BX_DECP_LOCATION_MEM);
          
          /* Got a mem-decoder at lower BAR.
               Someone wants to create a MEM decoder at the higher BAR.
               This is not allowed by PCI-X Spec (would need an even BAR).
          */
          BESTX_SPRINTF(errmsg,"Cannot create a memory decoder at an odd BAR (%lu). (File:%s,Line:%lu)",decoder,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_DECODER);
        }
      }
      else
      {
        assert(LocLow==MEM_DECODER);
        assert(decoder==0||decoder==2||decoder==4);
       
        if (val==BX_DECP_LOCATION_IO)
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==0||decoder==2||decoder==4);
          assert(val==BX_DECP_LOCATION_IO);

          /* 
            Someone wants to create an IO decoder at the lower BAR,
            which contains a memory decoder.
          */
   
          if (LocHigh==MEM_DECODER)
          {
            assert(LocLow==MEM_DECODER);
            assert(decoder==0||decoder==2||decoder==4);
            assert(val==BX_DECP_LOCATION_IO);
            assert(LocHigh==MEM_DECODER);

            /* Create IO decoder at lower BAR */
            /* low BAR */
            DB_CONFIG.value[decoder+4]=DEFAULT_IO_VAL;  
            DB_CONFIG.mask [decoder+4]=DEFAULT_IO_MASK;
            /* high BAR: zero-sized decoder */
            DB_CONFIG.value[decoder+1+4]=0;  
            DB_CONFIG.mask [decoder+1+4]=0; 
          }
          else
          {
            assert(LocLow==MEM_DECODER);
            assert(decoder==0||decoder==2||decoder==4);
            assert(val==BX_DECP_LOCATION_IO);
            assert(LocHigh==IO_DECODER);
            
            /* Create IO decoder at lower BAR */
            
            /* low BAR */
            DB_CONFIG.value[decoder+4]=DEFAULT_IO_VAL;  
            DB_CONFIG.mask [decoder+4]=DEFAULT_IO_MASK;
            
            /* high BAR: leave IO decoder unchanaged */
          }
        }
        else
        {
          assert(LocLow==MEM_DECODER);
          assert(decoder==0||decoder==2||decoder==4);
          assert(val==BX_DECP_LOCATION_MEM);

          if (LocHigh==IO_DECODER)
          {
            assert(SizeHigh); /* checked by BestXTConfigCheck() */

            /* We would overwrite an existing IO decoder at the higher BAR */
            BESTX_SPRINTF(errmsg,"Before creating a memory decoder at BAR pair (%lu,%lu), you first have to switch off the IO decoder at BAR %lu (File:%s,Line:%lu)",decoder,decoder+1,decoder+1,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_DECODER);
          }
          else
          {
            /* Create a MEM decoder with default size */
            DB_CONFIG.value[decoder+4]=DEFAULT_MEM_VAL_LO; 
            DB_CONFIG.mask [decoder+4]=DEFAULT_MEM_MASK_LO;  
            /* Higher BAR */
            DB_CONFIG.value[decoder+1+4]=DEFAULT_MEM_VAL_HI;
            DB_CONFIG.mask [decoder+1+4]=DEFAULT_MEM_MASK_HI; 
          }
        }
      }          
    }
  }    
        
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderBarPrefetchSet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 val)
{
  /* Sets prefetch propery for a BAR decoder */

  bx_int32 LocLow;

  BX_DECLARE_FUNCNAME("BestXTDecoderBarPrefetchSet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle,BX_PARAM_DECODER_STD,(bx_int32)BX_DECP_PREFETCH,val));
    BX_TRY(TDecoderSanityCheck(handle,decoder,&LocLow,NULL,NULL,NULL,NULL));

    if (LocLow==MEM_DECODER)
    {
      if (decoder%2)
      {
        /* BAR 1,3 or 5 */
        /* The higher BAR of a memory decoder "does not exist" */
        BESTX_SPRINTF(errmsg,"Cannot set prefetch bit at an odd BAR (%lu). (File:%s,Line:%lu)",decoder,__FILE__,__LINE__);
        (void) BestXLastErrorParamStringSet(handle,errmsg);
        BX_ERRETURN(BX_E_DECODER);
      }
      else
      {
        /* BAR 0,2,4 */
        /* 64 bit memory decoder */
        if (val)
        {
          /* set prefetch bit */
          DB_CONFIG.value[decoder+4] |= 1<<3;
        }
        else
        {
          /* clear prefetch bit */
          DB_CONFIG.value[decoder+4] &= ~(1<<3);
        }
      }
    }
    else
    {
      assert (LocLow==IO_DECODER);

      /* IO decoder at lower bar.
      Although there might be a zero-sized 
      memory decoder at the higher BAR,
      we do not allow to use that one */
    
      BESTX_SPRINTF(errmsg,"Cannot set prefetch bit at BAR %lu, because BAR %lu contains an IO decoder (File:%s,Line:%lu)",decoder,decoder%2?decoder-1:decoder,__FILE__,__LINE__);
      (void) BestXLastErrorParamStringSet(handle,errmsg);
      BX_ERRETURN(BX_E_DECODER);
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}


static bx_errtype BestXTDecoderBarPrefetchGet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 *val)
{
  /* Gets the prefetch property for a BAR decoder */

  bx_int32 LocLow;
  bx_int32 Prefetch;

  BX_DECLARE_FUNCNAME("BestXTDecoderBarPrefetchGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_STD,(bx_int32)BX_DECP_PREFETCH));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
    BX_TRY(TDecoderSanityCheck(handle,decoder,&LocLow,NULL,NULL,NULL,&Prefetch));

    if (LocLow==MEM_DECODER)
    {
      if (decoder%2)
      {
        /* BAR 1,3 or 5 */
        /* The higher BAR of a memory decoder "does not exist" */
        BESTX_SPRINTF(errmsg,"Cannot get prefetch bit of an odd BAR (%lu) (File:%s,Line:%lu)",decoder,__FILE__,__LINE__);
        (void) BestXLastErrorParamStringSet(handle,errmsg);
        BX_ERRETURN(BX_E_DECODER);
      }
      else
      {
        /* BAR 0,2,4 */
        /* 64 bit memory decoder */
        *val=Prefetch;
      }
    }
    else
    {
      /* IO decoder at lower bar.
         Although there might be a zero-sized 
         memory decoder at the higher BAR,
         we do not allow to use that one */
      BESTX_SPRINTF(errmsg,"Cannot get prefetch bit from BAR %lu, because BAR %lu contains an IO decoder (File:%s,Line:%lu)",decoder,decoder%2?decoder-1:decoder,__FILE__,__LINE__);
      (void) BestXLastErrorParamStringSet(handle,errmsg);
      BX_ERRETURN(BX_E_DECODER);
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

static bx_errtype BestXTDecoderBarLocationGet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 *val)
{
  /* Gets the location (BX_DEC_LOCATION) for a BAR deooder */

  bx_int32 LocLow,LocHigh;

  BX_DECLARE_FUNCNAME("BestXTDecoderBarLocationGet []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_STD,(bx_int32)BX_DECP_LOCATION));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
    BX_TRY(TDecoderSanityCheck(handle,decoder,&LocLow,&LocHigh,NULL,NULL,NULL));
    
    /* Check, wether decoder is off */
    BX_TRY(BestXTDecoderBarSizeGet(handle,decoder,val));
    if (*val==0)
    {
      /* decoder is off */
      *val=BX_DECP_LOCATION_OFF; 
    }
    else
    {
      *val=(decoder%2?LocHigh:LocLow);
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype BestXTDecoderBarSizeGet(
 *
 * Purpose      : 
 *---------------------------------------------------------------------------*/
static bx_errtype BestXTDecoderBarSizeGet(
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 *val)
{
  /* Gets the size of a BAR decoder */

  bx_int32 LocLow;
  bx_int32 SizeLow,SizeHigh;

  BX_DECLARE_FUNCNAME("BestXTDecoderBarSizeGet []");

  BX_TRY_VARS_NO_PROG;

  assert(decoder<=BX_DEC_BAR5);

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_STD,(bx_int32)BX_DECP_SIZE));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
    BX_TRY(TDecoderSanityCheck(handle,decoder,&LocLow,NULL,&SizeLow,&SizeHigh,NULL));
    
    *val=(decoder%2?SizeHigh:SizeLow);
  }
  
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype BestXTDecoderBaseAddrSet (
 *
 * Purpose      : 
 *---------------------------------------------------------------------------*/
static bx_errtype BestXTDecoderBaseAddrSet (
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32ptr pBaseLo,
  bx_int32ptr pBaseHi
  )
{
  BX_TRY_VARS_NO_PROG;
  BX_DECLARE_FUNCNAME("BestXTDecoderBaseAddrSet []");

  BX_TRY_BEGIN {
    bx_int32 value;
    bx_int32 offset;

    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_GEN, (bx_int32)decoder));

    /* standard decoder? */
    if (decoder >= BX_DEC_BAR0 && decoder <= BX_DEC_BAR5)
    {
      offset = 0x10UL + (bx_int32) (decoder-BX_DEC_BAR0) * sizeof(bx_int32);

      if ( (bx_int32)(decoder - BX_DEC_BAR0) % 2UL != 0 )
      {
        /* check previous decoder, this one might be grabbed */
        BX_TRY(BestXConfRegGet(handle, offset-4, &value));
        if (value & 0x7UL == 0x4UL) /* space64 decoder? -> error! */
        {
          BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32)decoder);
          BX_TRY_FAIL (BX_E_DEC_GRABBED);
        }
      }
      
      BX_TRY(BestXConfRegGet(handle, offset, &value));

      if (value & 1UL)          /* I/O decoder? */
      {
        if (pBaseLo)            /* do we want to set *pBaseLo? */
        {
          *pBaseLo |= 0x1UL;    /* set I/O bit */
          BX_TRY (BestXConfRegSet (handle, offset, *pBaseLo));
        }
      }
      else                      /* mem decoder */
      {
        switch (value & 0x6UL)
        {
         case 4UL:              /* 64bit decoder -> set hi address */

          if ( ((bx_int32)(decoder - BX_DEC_BAR0) % 2UL == 0) &&
               pBaseHi != NULL )
          {
            BX_TRY(BestXConfRegSet(handle, offset+4, *pBaseHi));
          }
          /* no break here intentionally */

         case 2UL:              /* below1meg */
          /* no break here intentionally */
         case 0UL:              /* space32 */
          if ( pBaseLo != NULL )
          {
            /* filter location/prefetch bits */
            *pBaseLo =  (*pBaseLo & ~0xFUL) | (value & 0xFUL);
            BX_TRY(BestXConfRegSet (handle, offset, *pBaseLo));
          }
          break;

         case 6UL:              /* something's wrong here */
          BX_TRY_FAIL (BX_E_INVALID_CASE);
        }
      }
    }
    else if (decoder == BX_DEC_EXPROM)
    {
      offset = 0x30UL;
      if (pBaseLo != NULL)
      {
        BX_TRY(BestXConfRegGet(handle, offset, &value));
        /* mask out <enable> bit */
        *pBaseLo = (*pBaseLo & ~0x1UL) | (value & 0x1UL);
        BX_TRY(BestXConfRegSet(handle, offset, *pBaseLo));
      }
    }
    else
    {
      /* other decoders do not have a base address */
      BX_TRY_FAIL (BX_E_TBD);   /* TODO: find error msg */
    }
  }

  BX_TRY_CATCH {
    /* enter error handling here */
  }

  return (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype BestXTDecoderBaseAddrGet (
 *
 * Purpose      : 
 *---------------------------------------------------------------------------*/
static bx_errtype BestXTDecoderBaseAddrGet (
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32ptr pBaseLo,
  bx_int32ptr pBaseHi
  )
{
  BX_TRY_VARS_NO_PROG;
  BX_DECLARE_FUNCNAME("BestXTDecoderBaseAddrGet []");

  BX_TRY_BEGIN 
  {
    bx_int32 value;
    bx_int32 offset;

    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_GEN, (bx_int32)decoder));

    /* standard decoder? */
    if (decoder >= BX_DEC_BAR0 && decoder <= BX_DEC_BAR5)
    {
      offset = 0x10UL + (bx_int32) (decoder-BX_DEC_BAR0) * sizeof(bx_int32);

      if ( (bx_int32)(decoder - BX_DEC_BAR0) % 2UL != 0 )
      {
        /* check previous decoder, this one might be grabbed */
        BX_TRY(BestXConfRegGet(handle, offset-4, &value));
        if (value & 0x7UL == 0x4UL) /* space64 decoder? -> error! */
        {
          BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32)decoder);
          BX_TRY_FAIL (BX_E_DEC_GRABBED);
        }
      }
      
      BX_TRY(BestXConfRegGet(handle, offset, &value));

      if (value & 1UL)          /* I/O decoder? */
      {
        if (pBaseLo)
        {
          *pBaseLo = value & ~1UL;      /* filter out I/O bit */
        }
      }
      else
      {
        switch (value & 0x6UL)
        {
         case 4UL:              /* 64bit decoder -> read hi address */

          if ( ((bx_int32)(decoder - BX_DEC_BAR0) % 2UL == 0) &&
               pBaseHi )
          {
            BX_TRY(BestXConfRegGet(handle, offset+4, pBaseHi));
          }
          /* no break here intentionally */

         case 2UL:              /* below1meg */
          /* no break here intentionally */
         case 0UL:              /* space32 */
          if (pBaseLo)
          {
            *pBaseLo = value & ~0xfUL; /* filter out location/prefetch bits */
          }
          break;

         case 6UL:              /* something's wrong here */
          if (pBaseLo)
          {
            *pBaseLo = value;   /* just pass it along... */
          }
          BX_TRY_FAIL (BX_E_INVALID_CASE);/* something is wrong here */
        }
      }
    }
    else if (decoder == BX_DEC_EXPROM)
    {
      offset = 0x30UL;
      BX_TRY(BestXConfRegGet(handle, offset, &value));
      if (pBaseLo)
      {
        *pBaseLo = value & ~1UL;        /* filter out enable bit */
      }
    }
    else
    {
      /* other decoders do not have a base address */
      BX_TRY_FAIL (BX_E_TBD);   /* TODO: find error msg */
    }
  }

  /* enter error handling here */

  return (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype TDecoderSanityCheck(
 *
 * Purpose      : 
 *---------------------------------------------------------------------------*/
static bx_errtype TDecoderSanityCheck(
  bx_handletype handle,
  bx_dectype BarDecoder,       /* IN: Bar0,..,Bar5 */
  bx_int32  *pLocationLow,     /* OUT: Bit 0 of lower BAR: IO or MEM */     
  bx_int32  *pLocationHigh,    /* OUT:                               */ 
  bx_int32  *pSizeLow,         /* OUT: size of lower BAR             */
  bx_int32  *pSizeHigh,        /* OUT:                               */ 
  bx_int32  *pPrefetch         /* OUT:                               */
)
{
  /* 
     This function is called from all property Set/Get() 
     functions for BARs at the beginning to make sure we have got
     a compliant config space.
     This function checks compliance of a BAR-pair
     ((BAR0,BAR1),(BAR2,BAR3) or (BAR4,BAR5)).
     The BAR-pair to check is identified via
     parameter BarDecoder.
     When BX_E_OK is returned, the properties hold
     in Config space (BX_DECP_LOCATION,BX_DECP_SIZE
     and BX_DECP_PREFETCH are returned.
     Passing NULL pointer is always OK.
     This function does no write accesses to config-space ! 
  */
     

  BX_DECLARE_FUNCNAME("TDecoderSanityCheck []");

  BX_TRY_VARS_NO_PROG;

  /* Return Values */
  bx_int32 LocationLow= 0xdeadbeaf;     
  bx_int32 LocationHigh=0xdeadbeaf;    /* only valid,if LocationLow=IO  */ 
  bx_int32 SizeLow=     0xdeadbeaf;
  bx_int32 SizeHigh=    0xdeadbeaf;    /* only valid,if LocationLow=IO  */ 
  bx_int32 Prefetch=    0xdeadbeaf;    /* only valid,if LocationLow=MEM */

  bx_int32 i,j;

  /* decoder is a valid BAR-decoder here */
  assert(BarDecoder<=BX_DEC_BAR5);

  /* Make BarDecoder the lower Bar of the Bar-pair.
     This is OK, because we do not check a specific 
     BAR decoder, but the corresponding decoder pair. 
  */

  if (BarDecoder%2)
  {
    /* start with an even BarDecoder (the lower one) */
    BarDecoder--;
  }

  assert(BarDecoder==0||BarDecoder==2||BarDecoder==4);

  BX_TRY_BEGIN
  {
    /* Bit 0 of lower BAR should always be Read-only: */
    /* Remark: 4-th DW is BAR0 in Config-space */
    if ((DB_CONFIG.mask[BarDecoder+4]&1)!=RO_BIT)
    {
      BESTX_SPRINTF(errmsg,"Mask of Bit 0 of BAR %lu must always be RO (File:%s,Line:%lu)",BarDecoder,__FILE__,__LINE__);
      (void) BestXLastErrorParamStringSet(handle,errmsg);
      BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
    }
    
    /* Value in Bit 0 determines Location */
    LocationLow=DB_CONFIG.value[BarDecoder+4]&1;

    if (LocationLow==IO_DECODER)
    {
      /* The lower BAR is a 32 Bit IO decoder */
      
      /* Bit 1 of lower BAR should be reserved (read-only zero): */
      /* Remark: 4-th DW is BAR0 in Config-space */
      if ((DB_CONFIG.mask [BarDecoder+4]&2)!=RO_BIT)
      {
        BESTX_SPRINTF(errmsg,"Mask of Bit 1 (reserved) of IO decoder at BAR %lu must be RO (File:%s,Line:%lu)",BarDecoder,__FILE__,__LINE__);
        (void) BestXLastErrorParamStringSet(handle,errmsg);
        BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
      }
      if ((DB_CONFIG.value [BarDecoder+4]&2)!=0)
      {
        BESTX_SPRINTF(errmsg,"Value of Bit 1 (reserved) of IO decoder at BAR %lu must be zero (File:%s,Line:%lu)",BarDecoder,__FILE__,__LINE__);
        (void) BestXLastErrorParamStringSet(handle,errmsg);
        BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
      }
      
      /* Determine size of IO decoder */
      i=2;  /* meaning of Bits 0 and 1 already checked above */
      while ((DB_CONFIG.mask[BarDecoder+4]&1<<i)==RO_BIT && i<32)
      {
        /* Bit i is read-only */

        /* Value should be zero: */
        if ((DB_CONFIG.value[BarDecoder+4]&1<<i)!=0)
        {
          BESTX_SPRINTF(errmsg,"Value of Bit %lu of IO decoder at BAR %lu must be zero, when mask is RO (File:%s,Line:%lu)",i,BarDecoder,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
        }
        
        i++;
      }

      /* Either i==32 (all bits read-only) or Bit i is writable */
      SizeLow = (i==32?0:i);

      assert(SizeLow!=1 && SizeLow<32);

      if (SizeLow==0)
      {
        BESTX_SPRINTF(errmsg,"IO decoder of size zero  (BAR %lu) is not Spec compliant (File:%s,Line:%lu)",BarDecoder,__FILE__,__LINE__);
        (void) BestXLastErrorParamStringSet(handle,errmsg);
        BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
      }

      assert(1<SizeLow && SizeLow<32);

      /* Make sure that the remaining Bits are writable too */
      for (j=i+1;j<32;j++)
      {
        if ((DB_CONFIG.mask[BarDecoder+4]&1<<j)==RO_BIT)
        {
          BESTX_SPRINTF(errmsg,"Mask of Bit %lu of IO decoder at BAR %lu must be RW too, when mask of Bit %lu is already RW (File:%s,Line:%lu)",j,BarDecoder,i,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }
      }

      /****************************************************/
      /* BarDecoder is a valid IO decoder of size SizeLow */
      /****************************************************/
    
      /* Lets look at the higher bar (BarDecoder+1).
         It should also be an IO decoder or a memory 
         decoder of size=0 (all bits RO)
       */

      /* Bit 0 of higher BAR should always be Read-only: */
      if ((DB_CONFIG.mask[BarDecoder+1+4]&1)!=RO_BIT)
      {
        /* Remark: 4-th DW is BAR0 in Config-space */
        BESTX_SPRINTF(errmsg,"Mask of Bit 0 of decoder at BAR %lu must be RO, when BAR %lu contains an IO decoder (File:%s,Line:%lu)",BarDecoder+1,BarDecoder,__FILE__,__LINE__);
        (void) BestXLastErrorParamStringSet(handle,errmsg);
        BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
      }

      /* Value in Bit 0 determines Location */
      LocationHigh=DB_CONFIG.value[BarDecoder+1+4]&1;

      if (LocationHigh==IO_DECODER)
      {
        /* The higher BAR is a 32 Bit IO decoder */

        /* Bit 1 of higher BAR should be reserved (read-only zero): */
        if ((DB_CONFIG.mask [BarDecoder+1+4]&2)!=RO_BIT)
        {
          /* Remark: 4-th DW is BAR0 in Config-space */
          BESTX_SPRINTF(errmsg,"Mask of Bit 1 (reserved) of IO decoder at BAR %lu must be RO (File:%s,Line:%lu)",BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }
        if ((DB_CONFIG.value [BarDecoder+1+4]&2)!=0)
        {
          /* Remark: 4-th DW is BAR0 in Config-space */
          BESTX_SPRINTF(errmsg,"Value of Bit 1 (reserved) of IO decoder at BAR %lu must be zero (File:%s,Line:%lu)",BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
        }
      
      
        /* Determine size of IO decoder */
        i=2;  /* meaning of Bits 0 and 1 already checked above */
        while ((DB_CONFIG.mask[BarDecoder+1+4]&1<<i)==RO_BIT && i<32)
        {
          /* Bit i is read-only */

          /* Value should be zero: */
          if ((DB_CONFIG.value[BarDecoder+1+4]&1<<i)!=0)
          {
            BESTX_SPRINTF(errmsg,"Value of Bit %lu of IO decoder at BAR %lu must be zero, when mask is RO (File:%s,Line:%lu)",i,BarDecoder+1,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
          }

          i++;
        }

        /* Either i==32 (all bits read-only) or Bit i is writable */
        SizeHigh = (i==32?0:i);

        assert(SizeHigh!=1 && SizeHigh<32);

        if (SizeHigh==0)
        {
          BESTX_SPRINTF(errmsg,"IO decoder of size zero  (BAR %lu) is not Spec compliant (File:%s,Line:%lu)",BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }

        assert(1<SizeHigh && SizeHigh<32);


        /* Make sure that the remaining Bits are writable too */
        for (j=i+1;j<32;j++)
        {
          if ((DB_CONFIG.mask[BarDecoder+1+4]&1<<j)==RO_BIT)
          {
            BESTX_SPRINTF(errmsg,"Mask of Bit %lu of IO decoder at BAR %lu must be RW too, when mask of Bit %lu is already RW (File:%s,Line:%lu)",j,BarDecoder+1,i,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
          }
        }

        /*******************************************************/
        /* BarDecoder+1 is a valid IO decoder of size SizeHigh */
        /*******************************************************/
    
      } /* LocationHigh==IO */
      else
      {
        /* The higer BAR is a MEM decoder.(and the lower a valid IO decoder)
           Because in PCI-X only 64 Bit memory decoders are 
           allowed, we allow this case only, if its all zero 
        */

        if (DB_CONFIG.mask[BarDecoder+1+4])
        {
          BESTX_SPRINTF(errmsg,"Mask of BAR %lu must be all zero, when BAR %lu contains an IO decoder (File:%s,Line:%lu)",BarDecoder+1,BarDecoder,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }
        if (DB_CONFIG.value[BarDecoder+1+4])
        {
          BESTX_SPRINTF(errmsg,"Value of BAR %lu must be all zero, when BAR %lu contains an IO decoder (File:%s,Line:%lu)",BarDecoder+1,BarDecoder,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
        }
              
        SizeHigh=0;
      } /* LocationHigh==MEM */
    } /* LocationLow==IO */
    else
    {
      assert(LocationLow==MEM_DECODER);

      /* There may be an IO decoder at the higher BAR and a zero-sized (mem) decoder at the lower BAR */

      LocationHigh = (((DB_CONFIG.mask [BarDecoder+4+1]&1)==RO_BIT && 
                       (DB_CONFIG.value[BarDecoder+4+1]&1)==1) ? IO_DECODER :MEM_DECODER );

      if (LocationHigh==IO_DECODER)
      {
        assert(LocationLow==MEM_DECODER);
        assert(LocationHigh==IO_DECODER);
        
        /* Assumption:
           IO decoder at the higher BAR and a zero-sized (mem) decoder at the lower BAR */
        if (DB_CONFIG.value[BarDecoder+4] || DB_CONFIG.mask[BarDecoder+4])
        {
          /* mask and value should be all zero */
          BESTX_SPRINTF(errmsg,"Value (0x%08lx) and Mask (0x%08lx) of BAR %lu must be all zero, when BAR %lu contains an IO decoder (File:%s,Line:%lu)",DB_CONFIG.value[BarDecoder+4],DB_CONFIG.mask[BarDecoder+4],BarDecoder,BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
        }

        SizeLow=0;

        /* Lower BAR has zero-sized (mem) decoder, that's OK.
           Now check the IO decoder at the higher BAR */
        /* The higher BAR should be a 32 Bit IO decoder */
      
        /* Bit 1 of lower BAR should be reserved (read-only zero): */
        /* Remark: 4-th DW is BAR0 in Config-space */
        if ((DB_CONFIG.mask [BarDecoder+1+4]&2)!=RO_BIT)
        {
          BESTX_SPRINTF(errmsg,"Mask of Bit 1 (reserved) of IO decoder at BAR %lu must be RO (File:%s,Line:%lu)",BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }
        if ((DB_CONFIG.value [BarDecoder+4]&2)!=0)
        {
          BESTX_SPRINTF(errmsg,"Value of Bit 1 (reserved) of IO decoder at BAR %lu must be zero (File:%s,Line:%lu)",BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
        }
      
        /* Determine size of IO decoder */
        i=2;  /* meaning of Bits 0 and 1 already checked above */
        while ((DB_CONFIG.mask[BarDecoder+1+4]&1<<i)==RO_BIT && i<32)
        {
          /* Bit i is read-only */

          /* Value should be zero: */
          if ((DB_CONFIG.value[BarDecoder+1+4]&1<<i)!=0)
          {
            BESTX_SPRINTF(errmsg,"Value of Bit %lu of IO decoder at BAR %lu must be zero, when mask is RO (File:%s,Line:%lu)",i,BarDecoder+1,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
          }
        
          i++;
        }

        /* Either i==32 (all bits read-only) or Bit i is writable */
        SizeHigh = (i==32?0:i);

        assert(SizeHigh!=1 && SizeHigh<32);

        if (SizeHigh==0)
        {
          BESTX_SPRINTF(errmsg,"IO decoder of size zero  (BAR %lu) is not Spec compliant (File:%s,Line:%lu)",BarDecoder+1,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }

        assert(1<SizeHigh && SizeHigh<32);

        /* Make sure that the remaining Bits are writable too */
        for (j=i+1;j<32;j++)
        {
          if ((DB_CONFIG.mask[BarDecoder+1+4]&1<<j)==RO_BIT)
          {
            BESTX_SPRINTF(errmsg,"Mask of Bit %lu of IO decoder at BAR %lu must be RW too, when mask of Bit %lu is already RW (File:%s,Line:%lu)",j,BarDecoder+1,i,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
          }
        }

        /****************************************************/
        /* BarDecoder is a valid IO decoder of size SizeHigh */
        /****************************************************/
      }
      else
      {
        assert(LocationLow==MEM_DECODER);
        assert(LocationHigh==MEM_DECODER);

        /* 
           The lower BAR is a 64 Bit MEM decoder and
           the higher BAR is also part of the Memory decoder.
        */

        /* Determine the decoder size: 
           We check each BAR individually and store 
           the sizes in SizeLow resp. SizeHigh.
           Then, we determine the final size from these 
           values and store them in SizeLow again.
         */

        /*******************/
        /* Check lower BAR */
        /*******************/

        i=1; /* Bit 0 already checked above to indicate MEM decoder */
        while ((DB_CONFIG.mask[BarDecoder+4]&1<<i)==RO_BIT && i<32)
        {
          /* Bit i is read-only */

          /* Value should be zero, if i>3 (Bits 1,2: Type-Bits; Bit 3: Prefetch-Bit) */
          if (i>3)
          {
            if ((DB_CONFIG.value[BarDecoder+4]&1<<i)!=0)
            {
              BESTX_SPRINTF(errmsg,"Value of Bit %lu of MEM decoder at BAR %lu must be zero, when mask is RO (File:%s,Line:%lu)",i,BarDecoder,__FILE__,__LINE__);
              (void) BestXLastErrorParamStringSet(handle,errmsg);
              BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
            }
          }

          i++;
        }

        /* Either i==32 (all bits read-only) or Bit i is writable */
      
        /* Minimum memory decoder size for PCI-X is 128 bytes */
        if (i<7)
        {      
          BESTX_SPRINTF(errmsg,"Minimum MEM decoder size for PCI-X is 128 bytes. Your MEM decoder at BAR %lu has size %lu bytes (File:%s,Line:%lu)",BarDecoder,1<<i,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }

        /* Make sure that the remaining Bits are writable too */
        for (j=i+1;j<32;j++) /* maybe a NOP */
        {
          if ((DB_CONFIG.mask[BarDecoder+4]&1<<j)==RO_BIT)
          {
            BESTX_SPRINTF(errmsg,"Mask of Bit %lu of MEM decoder at BAR %lu must be RW too, when mask of Bit %lu is already RW (File:%s,Line:%lu)",j,BarDecoder,i,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
          }
        }

        /* Check Type (Bits 1 and 2): value of 11 is reserved, i.e. disallowed */      
        if ((DB_CONFIG.value[BarDecoder+4]&6)==6)
        {
          BESTX_SPRINTF(errmsg,"MEM decoder at BAR %lu has invalid (reserved) value '11' at type bits 1 and 2 (File:%s,Line:%lu)",BarDecoder,__FILE__,__LINE__);
          (void) BestXLastErrorParamStringSet(handle,errmsg);
          BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
        }
      
        /* We do not check for compliance of these bits with
           value of Bit 0 here ! 
           BX_DECP_LOCATION is only determined by value of Bit 0!
           Maybe this needs to be enhanced/changed. */

        assert(i>=7 && i<=32);

        SizeLow=i; /* temporary size for lower BAR */
        /* This is not yet the final size of the 64 bit memory decoder here,
           but ca. the size of a 32-bit PCI memory decoder ! 
         */

        /********************/
        /* Check higher BAR */
        /********************/

        i=0; /* Location,prefetch and type bits are in lower BAR only ! */
        while ((DB_CONFIG.mask[BarDecoder+1+4]&1<<i)==RO_BIT && i<32)
        {
          /* Bit i is read-only */

          /* Value should be zero */
          if ((DB_CONFIG.value[BarDecoder+1+4]&1<<i)!=0)
          {
            BESTX_SPRINTF(errmsg,"Value of Bit %lu of MEM decoder at upper BAR %lu must be zero, when mask is RO (File:%s,Line:%lu)",i,BarDecoder+1,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_VALUE_INVALID);
          }

          i++;
        }

        /* Either i==32 (all bits read-only) or Bit i is writable */

        /* Make sure that the remaining Bits are writable too */
        for (j=i+1;j<32;j++) /* maybe a NOP */
        {
          if ((DB_CONFIG.mask[BarDecoder+1+4]&1<<j)==RO_BIT)
          {
            BESTX_SPRINTF(errmsg,"Mask of Bit %lu of MEM decoder at upper BAR %lu must be RW too, when mask of Bit %lu is already RW (File:%s,Line:%lu)",j,BarDecoder+1,i,__FILE__,__LINE__);
            (void) BestXLastErrorParamStringSet(handle,errmsg);
            BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
          }
        }
     
        assert(i<=32);

        SizeHigh=i; /* temporary size for higher BAR */
        /* This is not yet the final size of the 64 bit memory decoder here.
         */
      
        /********************************************************************/
        /********************************************************************/
        /********************************************************************/

        /* Here we have the two sizes SizeLow and SizeHigh.
           From these we determine the final size of the 64 bit memory decoder,
           and store it in SizeLow again.
           Caution: size==32 means all bits read-only.
                    size==0  means all bits writable (SizeHigh only)
         */

        assert(SizeLow>=7 && SizeLow<=32);
        assert(SizeHigh<=32);

        if (SizeLow<32 && SizeHigh==0)
        {
          /* Decoder is below 4GB.
             SizeLow is already the final decoder size in [7,..,31]
          */
        }
        else
        {
          if (SizeLow==32 && SizeHigh<32)
          {
            /* Decoder is above or equal 4GB in [32,..,63] */
            SizeLow=SizeHigh+32;
          }
          else
          {
            if (SizeLow==32 && SizeHigh==32)
            {
              /* Decoder is off */
              SizeLow=0;
            }
            else
            {
              /* all other cases are invalid */
              BESTX_SPRINTF(errmsg,"Cannot determine the size of MEM decoder at BAR pair (%lu,%lu). Please check mask bits. (File:%s,Line:%lu)",BarDecoder,BarDecoder+1,__FILE__,__LINE__);
              (void) BestXLastErrorParamStringSet(handle,errmsg);
              BX_ERRETURN(BX_E_CONFIG_MASK_INVALID);
            }
          }
        }

        SizeHigh=0; /* SizeLow contains valid 64-bit size ! */
        
        /***********************************************************************/
        /* Got a valid 64 bit memory decoder with size SizeLow */
        /***********************************************************************/

        /* Get the Prefetch bit (bit 3 of lower BAR): */
        Prefetch = (DB_CONFIG.value[BarDecoder+4]&8 ? 1 : 0);

      } /* LocationHigh==MEM */
    }  /* LocationLow==MEM */
  } /* TRY-BLOCK */
  
  /* set return values: */
  if (pLocationLow)  *pLocationLow =LocationLow;
  if (pLocationHigh) *pLocationHigh=LocationHigh; 
  if (pSizeLow)      *pSizeLow=SizeLow;
  if (pSizeHigh)     *pSizeHigh=SizeHigh;         /* only valid, if LocationLow=IO */
  if (pPrefetch)     *pPrefetch=Prefetch;         /* only valid, if LocationLow=MEM */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXTConfigPUCodeGet     
 *
 * Purpose: This function is called from BestXLoaderProg() and assembles
 *          the loadercode of the config-space for Mephisto's powerup-loader.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTConfigPUCodeGet(
  bx_handletype handle,
  bx_int8ptr  Code,    /* OUT:   Powerup LoaderCode         */
  bx_int32ptr Length   /* INOUT: LoaderCode length in bytes */
)
{
  BX_DECLARE_FUNCNAME("BestXTConfigPUCodeGet []");

  bx_int32 RespectBios; /* Board property */
  
  BX_TRY_VARS_NO_PROG;

  /* Organisation of config space (32 DWs):
     DW  0-15: Standard PCI config space header (0x0-0x3c)
     DW 16-23: Programming registers; only accessible from PCI bus (0x40-0x5c)
     DW 24-31: PCIX part (0x60-0x7c):
               Status- and Command register, 
               Power Management, Message control 
  */
  
  /* Config space starts with Vendor ID */
  bx_int32 CfgValAddress =BX_REG_DEV_VEN_ID_REG;
  bx_int32 CfgMaskAddress=BX_REG_DEV_VEN_ID_MASK_REG;
  
  bx_int32 i,val,defval,mask;

  bx_int32 ConfigSize; /* In DWs */
  
#ifndef DOSW32
  __btmp=0;__env; /* avoid compiler warnings */
#endif
  
  assert(Code);
  assert(Length);

  BX_TRY_BEGIN
  {
    /* Config space contains 32 resp 36 DWs */
    assert(BX_CTCFG_MEMDEPTH==32 && BX_CTCFG_MEMDEPTHMAX==36);

    ConfigSize=(BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX);

    for (i=0;i<ConfigSize;i++)
    {
      /* Write mask and value for the i-th DW in config-space */
    
      if (16<=i && i<=23)
      {
        /* The 8 DWs (16-23) from offset 0x40 to 0x5c are accessible only
           from PCI port and contain the programming registers.
           Skip this range:
        */
        continue;
      }

      /* Get value and mask from host DB */
      BX_TRY(BestXConfRegGet(handle,i*4,&val));
      BX_TRY(BestXConfRegMaskGet(handle,i*4,&mask));

      BX_TRY(BestXBoardGet(handle,BX_BOARD_RESPECTBIOS,&RespectBios));
      if (RespectBios)
      {
        /* This is the default */

        /* Set the RW-bits 
          (i.e. BIOS bits, i.e. where mask equals 1) to default */
        
        BX_TRY(BestXConfRegDefaultGet(handle,i*4,&defval));
        val = mask & defval | ~mask & val; 
      }
      else
      {
        /* Maybe system has no BIOS, write values unchanged to card */
      }

      /* Bit 7 of header type register determines, wether we got a multifinction device or not.
         This information needs to be written into BX_REG_GEN_DEC_PROP_REG (bit2)*/
      if (i==3 && BestXHasFaust(handle))
      {
        if (val & 1<<23)
        {
          /* Multi function device, set bit 2, enable multi function decoder */
          BX_LOADERCMD_MASKWRITEWORD(Code+(*Length),BX_REG_GEN_DEC_PROP_REG,1<<2,1<<2,*Length)
        }
        else
        {
          /* Single function device, clear bit 2, disable multi function decoder */
          BX_LOADERCMD_MASKWRITEWORD(Code+(*Length),BX_REG_GEN_DEC_PROP_REG,0,1<<2,*Length)
        }
      }

      BX_LOADERCMD_WRITELONG(Code+(*Length),CfgMaskAddress,mask,*Length)
      BX_LOADERCMD_WRITELONG(Code+(*Length),CfgValAddress,val,*Length)

      /* Next DW */
      CfgMaskAddress+=4;
      CfgValAddress+=4;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTDecoderAllSet
 *
 * Purpose: convenience function 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXTDecoderAllSet( 
  bx_handletype   handle,
  bx_dectype      dec,
  bx_int32        location,
  bx_int32        size,
  bx_int32        prefetch,
  bx_int32        baselo,
  bx_int32        basehi,
  bx_int32        resource,
  bx_int32        resbase,
  bx_int32        ressize,
  bx_int32        compare
)
{
  bx_int32 val;
  BX_DECLARE_FUNCNAME("BestXTDecoderAllSet");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_LOCATION,location));
    BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_SIZE,size));
    if (location==BX_DECP_LOCATION_MEM)
    {
      BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_PREFETCH,prefetch));
    }
    BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_RESOURCE,resource));
    BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_RESBASE,resbase));
    BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_RESSIZE,ressize));
    BX_TRY(BestXTDecoderSet(handle,dec,BX_DECP_COMPARE,compare));

    /* Special handling for BARs */
    if (dec<=BX_DEC_BAR5 && size)
    {
      /* Set Base address */
      if (location==BX_DECP_LOCATION_IO)
      {
        assert(size<32);
        /* Just to be sure, cut off any low part */
        baselo &= ~((1<<size)-1); /* size > 0 !!!! */
        BX_TRY(BestXConfRegGet(handle,0x10+4*((bx_int32)dec),&val));
        BX_TRY(BestXConfRegSet(handle,0x10+4*((bx_int32)dec),val|baselo));
      }
      else
      {
        assert(location==BX_DECP_LOCATION_MEM);
        assert(size<64 && size>6);

        /* make dec even (the lower BAR) */
        dec -= ((bx_int32)dec)%2;

        if (size<32)
        {
          /* At least bit 31 of lower BAR is RW */

          /* Just to be sure, cut off any low part   */
          baselo &= ~((1<<size)-1);
          BX_TRY(BestXConfRegGet(handle,0x10+4*((bx_int32)dec),&val));
          BX_TRY(BestXConfRegSet(handle,0x10+4*((bx_int32)dec),val|baselo));

          /* The higher BAR is all RW */
          BX_TRY(BestXConfRegSet(handle,0x10+4*(1+(bx_int32)dec),basehi));
        }
        else
        {
          assert(32<=size);
          assert(size<64);

          /* lower BAR is all RO, so it remains unchanged */
          
          /* higher BAR: at least bit 31 (63) is RW */
          /* Just to be sure, cut off any low part  */
          basehi &= ~((1<<(size-32))-1);
          BX_TRY(BestXConfRegSet(handle,0x10+4*(1+(bx_int32)dec),basehi));
        }
      }
    }    
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*****************************************************************************
 * used by GUI (not documented publicly)
 *****************************************************************************/
/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXTDecoderInfo (
 *
 * Purpose      : get decoder info based on dec prop
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXTDecoderInfo (
  bx_handletype handle,
  bx_dectype decoder,
  bx_decproptype prop,
  const bx_param_infotype ** pParam
  )
{
  BX_DECLARE_FUNCNAME("BestXTDecoderInfo");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {

    BX_TRY (BestXParamInfoGet (handle,
                               BX_PARAM_DECODER_GEN,
                               decoder,
                               pParam,
                               BX_ENUM_SEARCH));

    /* now we have access to the decoder props */
    BX_TRY (BestXParamInfoGet (handle,
                               (*pParam)->group,
                               prop,
                               pParam,
                               BX_ENUM_SEARCH));

  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * CLI bx_errtype BestXTDecoderAddrRead (
 *
 * Purpose      : read base addresses from decoder
 *---------------------------------------------------------------------------*/
CLI bx_errtype EXPORT BestXTDecoderAddrRead (
  bx_handletype handle,
  bx_dectype decoder,
  bx_int32 * pAddress,
  bx_int32 * pAddressHi
  )
{
  BX_DECLARE_FUNCNAME("BestXTDecoderAddrRead []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    bx_int32 value;
    bx_int32 offset;

    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_DECODER_GEN, (bx_int32)decoder));
    BX_TRY_FCT_PARAM_NULL_POINTER (pAddress);
    BX_TRY_FCT_PARAM_NULL_POINTER (pAddressHi);

    *pAddress = *pAddressHi = 0x0UL; /* initialize anyway */

    /* standard decoder? */
    if (decoder >= BX_DEC_BAR0 && decoder <= BX_DEC_BAR5)
    {
      offset = 0x10UL + (bx_int32) (decoder-BX_DEC_BAR0) * sizeof(bx_int32);

      if ( (bx_int32)(decoder - BX_DEC_BAR0) % 2UL != 0 )
      {
        /* check previous decoder, this one might be grabbed */
        BX_TRY(BestXConfRegRead(handle, offset-4, &value));
        if (value & 0x7UL == 0x4UL) /* space64 decoder? -> error! */
        {
          BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32)decoder);
          BX_TRY_FAIL (BX_E_DEC_GRABBED);
        }
      }
      
      BX_TRY(BestXConfRegRead(handle, offset, &value));

      if (value & 1UL)          /* I/O decoder? */
      {
        *pAddress = value & ~1UL;       /* filter out I/O bit */
      }
      else
      {
        switch (value & 0x6UL)
        {
         case 4UL:              /* 64bit decoder -> read hi address */

          if ( (bx_int32)(decoder - BX_DEC_BAR0) % 2UL == 0 )
          {
            BX_TRY(BestXConfRegRead(handle, offset+4, pAddressHi));
          }
          /* no break here intentionally */

         case 2UL:              /* below1meg */
          /* no break here intentionally */
         case 0UL:              /* space32 */
          *pAddress = value & ~0xfUL; /* filter out location/prefetch bits */
          break;

         case 6UL:              /* somethings wrong here */
          *pAddress = value;    /* just pass it along... */
          BX_TRY_FAIL (BX_E_INVALID_CASE);/* something is wrong here */
        }
      }
    }
    else if (decoder == BX_DEC_EXPROM)
    {
      offset = 0x30UL;
      BX_TRY(BestXConfRegRead(handle, offset, &value));
      *pAddress = value & ~1UL; /* filter out enable bit */
    }

  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTConfigProg
 *
 * Purpose: Program config-space to card 
 *---------------------------------------------------------------------------*/
bx_errtype BestXTConfigProg (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXCTConfigProg [tconfigprog]");

  BX_TRY_VARS_NO_PROG;

  bx_int32 respectBios;    /* Board property      */
  bx_int32 MasterEnable;   /* Ri generic property */
  bx_int32 i;
  bx_int32 ConfigSize;

  BX_TRY_BEGIN
  {
    bx_dbctcfgtype cfgCopy;
    BX_TRY(BestXBoardGet(handle,BX_BOARD_RESPECTBIOS,&respectBios));
    BX_TRY(BestXRIGenGet(handle,BX_RIGEN_MENABLE,&MasterEnable));

    ConfigSize=(BestXHasMephisto(handle)?BX_CTCFG_MEMDEPTH:BX_CTCFG_MEMDEPTHMAX);

    if (respectBios)
    {
      /* read values */
      BX_TRY (ConfigValueRead ( handle, &(cfgCopy.value[0]) ));
      BX_TRY (ConfigMaskRead ( handle, &(cfgCopy.mask[0]) ));

      /* now, modify copy according to original mask */
      for (i=0; i< ConfigSize; i++)  /* this counts DWords! */
      {
        if (i==0x10)
        {
          i = 0x18;             /* exclude user configspace */
        }
        
        /* take rw-bit value from card and ro-bit value from db */
        cfgCopy.value[i] =
        cfgCopy.value[i]   &   DB_CONFIG.mask[i] |   /* RW part (HW) */
        DB_CONFIG.value[i] & ~(DB_CONFIG.mask[i]);   /* RO part (SW) */
      }

      if (MasterEnable)
      {
        /* set master enable bit in config space of host DB */
        cfgCopy.value[1] |= 0x4;      
      }

      /* now program values down to the card */
      BX_TRY (ConfigValueProg ( handle, &(cfgCopy.value[0]) ));
    }
    else
    {
      if (MasterEnable)
      {
        /* set master enable bit in config space of host DB */
        DB_CONFIG.value[1] |= 0x4;      
      }

      /* ConfigValueProg() modifies its parameter, so we have to copy here */
      memcpy(cfgCopy.value,DB_CONFIG.value,ConfigSize*sizeof(bx_int32));
      BX_TRY (ConfigValueProg (handle,cfgCopy.value));
    }
 
    /* mask always from DB !!! */
    /* ConfigMaskProg() modifies its parameter, so we have to copy here */
    memcpy(cfgCopy.mask,DB_CONFIG.mask,ConfigSize*sizeof(bx_int32));
    BX_TRY (ConfigMaskProg (handle,cfgCopy.mask));

  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTConfigRead
 *
 * Purpose: Read current config-space from card to host memory
 *---------------------------------------------------------------------------*/
bx_errtype BestXTConfigRead (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXTConfigRead [tconfigread]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* read values */
    BX_TRY (ConfigValueRead ( handle, &(DB_CONFIG.value[0]) ));
    BX_TRY (ConfigMaskRead ( handle, &(DB_CONFIG.mask[0]) ));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTDecoderProg
 *
 * Purpose: Program decoder-props from host memory to card or to PU code
 *---------------------------------------------------------------------------*/
bx_errtype BestXTDecoderProg (
  bx_handletype handle,
  bx_int8ptr LoaderCode,
  bx_int32ptr LoaderLength
)
{
  BX_DECLARE_FUNCNAME("BestXTDecoderProg []");

  BX_TRY_VARS_NO_PROG;
   
  /*
     This programs all decoder properties, 
     WHICH HAVE NO REPRESENTATION IN CONFIG SPACE,
     (i.e. RESOURCE,RESBASE,RESSIZE,COMPARE)
     from host to card. All other decoder properties 
     are implicitely programmed with the config space 
     (BestXCTConfigProg()).
  */
  
  /* Decoder properties for lower + higher BAR */
  bx_int32 PropLocation[2]; 
  bx_int32 PropResource[2]; 
  bx_int32 PropResSize[2];  
  bx_int32 PropResBase[2]; 
  bx_int32 PropSize[2]; 
  bx_int32 PropCompare[2];

  /* There is one general decoder enable register,
     which allows to enable each decoder individually */
  bx_int32 DecoderEnable=0; 

  /* These values are programmed for each BAR and the Expansion ROM decoder */
  bx_int16 DecoderControl; /* Type,resource,properties of a decoder         */
  bx_int32 DecoderResSize; /* Size of the internal resource                 */
  bx_int32 DecoderResBase; /* Offset for the internal resource              */

  bx_int32 Bar,b,i;          
  
  BX_TRY_BEGIN
  {
    /**************************************************************************
     * STD/BAR decoders 
     **************************************************************************/

    /* Handle all pairs of BARs */
    for (Bar=0;Bar<6;Bar+=2)
    {
      /* Get all properties from both BARs */
      
      /* Lower BAR */
      BX_TRY(BestXTDecoderGet(handle,Bar,BX_DECP_LOCATION,PropLocation+0));
      BX_TRY(BestXTDecoderGet(handle,Bar,BX_DECP_RESOURCE,PropResource+0));
      BX_TRY(BestXTDecoderGet(handle,Bar,BX_DECP_RESBASE,PropResBase+0));
      BX_TRY(BestXTDecoderGet(handle,Bar,BX_DECP_RESSIZE,PropResSize+0));
      BX_TRY(BestXTDecoderGet(handle,Bar,BX_DECP_SIZE,PropSize+0));
      BX_TRY(BestXTDecoderGet(handle,Bar,BX_DECP_COMPARE,PropCompare+0));

      /* Higher BAR */
      BX_TRY(BestXTDecoderGet(handle,Bar+1,BX_DECP_LOCATION,PropLocation+1));
      BX_TRY(BestXTDecoderGet(handle,Bar+1,BX_DECP_RESOURCE,PropResource+1));
      BX_TRY(BestXTDecoderGet(handle,Bar+1,BX_DECP_RESBASE,PropResBase+1));
      BX_TRY(BestXTDecoderGet(handle,Bar+1,BX_DECP_RESSIZE,PropResSize+1));
      BX_TRY(BestXTDecoderGet(handle,Bar+1,BX_DECP_SIZE,PropSize+1));
      BX_TRY(BestXTDecoderGet(handle,Bar+1,BX_DECP_COMPARE,PropCompare+1));
 
      /* check if ResourceSize<=DecoderSize */
      for(i=0;i<2;i++)
      {
        if(  (PropResSize[i] > PropSize[i]) 
             && (PropResSize[i] > 0) 
             && (PropSize[i]    > 0)
          ) 
        {
          PropResSize[i]=PropSize[i];
          BX_TRY(BestXTDecoderSet(handle,Bar+i,BX_DECP_RESSIZE,PropResSize[i]));
        }
      }

      if ( PropLocation[0]==BX_DECP_LOCATION_MEM)
      {
        /* Set up a 64 bit memory decoder */

        /* The decoder enable register (one for all decoders)
           enables decoders individually */
        DecoderEnable |= (PropSize[0] ? (0x3<<Bar):0x0); /* Enable e.g. STD0, STD1 */
        
        /* The decoder control register (one for each decoder)
           holds info about the resource,the decoder type and 
           the behavior (dac,fixed behavior) */
        DecoderControl = (PropResource[0]==BX_DECP_RESOURCE_GEN ? 
                          0x1 /*DataGen*/ : 0 /*MEM*/);
        DecoderControl |= 0x20; /* enable decoding of memory cmds */
        DecoderControl |= 0x40; /* enable decoding of dac    cmds */
        
        /* set the compare bit on HW if enabled in bar0 | bar2 | bar4 */
        if(PropCompare[0])
        {
          DecoderControl |= 0x00000080;
        }
 
        /* The decoder size register holds the size of the internal resource */
        DecoderResSize = ~((1<<PropResSize[0])-1); /* =0xffffff00 for PropResSize=8, i.e. 256 bytes */

        /* The decoder base register holds the offset for the internal resource */
        DecoderResBase = PropResBase[0];
      
        /* Control */
        if (LoaderCode)
        {
          /* Setting Powerup loader code */
          BX_LOADERCMD_WRITEWORD(LoaderCode+*LoaderLength,DecAddr[Bar].Control,DecoderControl,*LoaderLength)
          BX_LOADERCMD_WRITEWORD(LoaderCode+*LoaderLength,DecAddr[Bar+1].Control,DecoderControl,*LoaderLength)
        }
        else
        {
          /* Programming card */
          BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar].Control,sizeof(bx_int16),DecoderControl));
          BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar+1].Control,sizeof(bx_int16),DecoderControl));
        }
        
        /* ResBase */
        if (LoaderCode)
        {
          /* Setting Powerup loader code */
          BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[Bar].ResBase,DecoderResBase,*LoaderLength)
          BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[Bar+1].ResBase,DecoderResBase,*LoaderLength)
        }
        else
        {
          /* Programming card */
          BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar].ResBase,sizeof(bx_int32),DecoderResBase));
          BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar+1].ResBase,sizeof(bx_int32),DecoderResBase));
        }
 
        /* ResSize, must be in {2,..,24} */
        if (LoaderCode)
        {
          /* Setting Powerup loader code */
          BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[Bar].ResSize,DecoderResSize,*LoaderLength)
          BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[Bar+1].ResSize,DecoderResSize,*LoaderLength)
        }
        else
        {
          /* Programming card */
          BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar].ResSize,sizeof(bx_int32),DecoderResSize));
          BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar+1].ResSize,sizeof(bx_int32),DecoderResSize));
        }
      }
      else
      {
        /* Setup two IO decoders resp off decoders:
           Maybe any of: {(IO,Off),(IO,IO),(Off,IO),(Off,Off)}
        */

        assert(PropLocation[0]==BX_DECP_LOCATION_IO || PropLocation[0]==BX_DECP_LOCATION_OFF);
        
        /* We have two separate decoders */
        
        for (b=0;b<2;b++) /* for lower and higher BAR */
        {
          /* The decoder enable register (one for all decoders)
             enables decoders individually */
          DecoderEnable |= (PropSize[b]?(0x1<<(Bar+b)):0); /* Enable BAR */
        
          /* The decoder control register (one for each decoder)
             holds info about the resource,the decoder type and 
             the behavior (dac,fixed behavior) */
          switch (PropResource[b])
          {
            case BX_DECP_RESOURCE_GEN:  /* data generator */ 
              DecoderControl =1;
              break;          
            case BX_DECP_RESOURCE_CFG:  /* config space */ 
              /* 19.8.2003 added for SR
                 Caution: BX_DECP_RESOURCE_CFG works only with IO decoder.
                 Problems with MEM decoders (WAITS)
              */
              DecoderControl =0xa; /* resource config space and fixed behavior (bit 3) */
              break;
            default:  /* data memory */ 
              DecoderControl =0;
          }

          DecoderControl |= (PropLocation[b]==BX_DECP_LOCATION_IO ? 0x10:0x0); /* enable decoding of IO cmds */

          /* set the compare bit */
          if(PropCompare[b])
          {
            DecoderControl |= 0x00000080;
          }
          
          /* The decoder size register holds the size of the internal resource */
          DecoderResSize = ~((1<<PropResSize[b])-1); /* =0xffffff00 for ResSize=8, i.e. 256 bytes */

          /* The decoder base register holds the offset for the internal resource */
          DecoderResBase = PropResBase[b];
      
          /* Program control,resbase and ressize (in {2,..,24}) register to HW */
          if (LoaderCode)
          {
            /* Setting Powerup loader code */
            BX_LOADERCMD_WRITEWORD(LoaderCode+*LoaderLength,DecAddr[Bar+b].Control,DecoderControl,*LoaderLength)
            BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[Bar+b].ResBase,DecoderResBase,*LoaderLength)
            BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[Bar+b].ResSize,DecoderResSize,*LoaderLength)
          }
          else
          {
            /* Programming card */
            BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar+b].Control,sizeof(bx_int16),DecoderControl));
            BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar+b].ResBase,sizeof(bx_int32),DecoderResBase));
            BX_TRY(BestXDirectRegWrite(handle,DecAddr[Bar+b].ResSize,sizeof(bx_int32),DecoderResSize));
          }
        }
      }
    }
  
    /************************************************************************** 
     * Expansion ROM decoder
     **************************************************************************/

    /* Get all properties for expansion ROM decoder */
    
    BX_TRY(BestXTDecoderGet(handle,BX_DEC_EXPROM,BX_DECP_RESOURCE,PropResource+0));
    BX_TRY(BestXTDecoderGet(handle,BX_DEC_EXPROM,BX_DECP_RESBASE,PropResBase+0));
    BX_TRY(BestXTDecoderGet(handle,BX_DEC_EXPROM,BX_DECP_RESSIZE,PropResSize+0));
    BX_TRY(BestXTDecoderGet(handle,BX_DEC_EXPROM,BX_DECP_SIZE,PropSize+0));

    /* The decoder enable register (one for all decoders)
       enables decoders individually */
    DecoderEnable |= (PropSize[0]?1<<6:0); /* Enable/disable exp rom decoder */

    /* The decoder control register (one for each decoder)
       holds info about the resource,the decoder type and 
       the behavior (dac,fixed behavior) */
    
    if (PropResource[0]==BX_DECP_RESOURCE_GEN)
    {
      DecoderControl = 0x1;
    }
    else if (PropResource[0]==BX_DECP_RESOURCE_FLASH)
    {
      DecoderControl = 0xb;
    }
    else 
    {
      DecoderControl = 0x0 /* MEM */;
    }

    if (PropResource[0]==BX_DECP_RESOURCE_FLASH)
    {  /* set fix size for FLASH resource */
       DecoderResSize = ~(BX_EXPROM_IMAGE_SIZE -1); 
    }
    else
    {
        /* The decoder size register holds the size of the internal resource */
        DecoderResSize = ~((1<<PropResSize[0])-1); /* =0xffffff00 for ResSize=8, i.e. 256 bytes */
    }

    if (PropResource[0]==BX_DECP_RESOURCE_FLASH)
    {  /* set fix base for FLASH resource */
       DecoderResBase = BX_EXPROM_IMAGE; 
    }
    else
    {
       /* The decoder base register holds the offset for the internal resource */
       DecoderResBase = PropResBase[0];
    }
    /* Program control,resbase and ressize (in {2,..,24}) register to HW */
    if (LoaderCode)
    {
      /* Setting Powerup loader code */
      BX_LOADERCMD_WRITEWORD(LoaderCode+*LoaderLength,DecAddr[BX_DEC_EXPROM].Control,DecoderControl,*LoaderLength)
      BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[BX_DEC_EXPROM].ResBase,DecoderResBase,*LoaderLength)
      BX_LOADERCMD_WRITELONG(LoaderCode+*LoaderLength,DecAddr[BX_DEC_EXPROM].ResSize,DecoderResSize,*LoaderLength)
    }
    else
    {
      /* Programming card */
      BX_TRY(BestXDirectRegWrite(handle,DecAddr[BX_DEC_EXPROM].Control,sizeof(bx_int16),DecoderControl));
      BX_TRY(BestXDirectRegWrite(handle,DecAddr[BX_DEC_EXPROM].ResBase,sizeof(bx_int32),DecoderResBase));
      BX_TRY(BestXDirectRegWrite(handle,DecAddr[BX_DEC_EXPROM].ResSize,sizeof(bx_int32),DecoderResSize));
    }

    /**************************************************************************
     * Config decoder
     **************************************************************************/

    /* Get properties for config decoder */
    BX_TRY(BestXTDecoderGet(handle,BX_DEC_CONFIG,BX_DECP_SIZE,PropSize+0));

    /* The decoder enable register (one for all decoders)
       enables decoders individually */
    DecoderEnable |= (PropSize[0]?1<<8:0); /* Enable config decoder */

    /************************************************************************** 
     * Requester Target decoder
     **************************************************************************/

    /* Get properties for requester decoder */
    BX_TRY(BestXTDecoderGet(handle,BX_DEC_RT,BX_DECP_SIZE,PropSize+0));

    /* The decoder enable register (one for all decoders)
       enables decoders individually */
    DecoderEnable |= (PropSize[0]?1<<7:0); /* Enable config decoder */

    /* Now program the general decoder enable register */
    if (LoaderCode)
    {
      /* Setting Powerup loader code */
      BX_LOADERCMD_WRITEWORD(LoaderCode+*LoaderLength,BX_REG_DEC_EN_REG,DecoderEnable,*LoaderLength)
    }
    else
    {
      /* Programming card */
      BX_TRY(BestXSyncRegWrite(handle,
                             
                             BX_REG_DEC_EN_STATUS_REG,
                             sizeof(bx_int16),
                             0x1,
                             0x1,
                             
                             BX_REG_DEC_EN_REG,
                             sizeof(bx_int16),
                             0x1FF,
                             DecoderEnable,
                             
                             BX_REG_DEC_EN_SREG,
    
                             BX_REG_DEC_EN_FORCE_REG,
                             sizeof(bx_int16),
                             0 /* no soft force */));
                              
    }

    /***********************************************************************************/
    /* HW bugfix: The DAC enable bits must be cleared too, when decoder is disabled !! */
    /***********************************************************************************/
    if (!(DecoderEnable & 1<<0))
    {
      /* Disable STD0 by clearing the DAC-enable bit too */
      if (LoaderCode)
      {
        BX_LOADERCMD_MASKWRITEWORD(LoaderCode+*LoaderLength,BX_REG_STD0_CTRL_REG,0,1<<6,*LoaderLength)
      }
      else
      {
        BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD0_CTRL_REG,sizeof(bx_int16),1<<6,0));
      }
    }
    if (!(DecoderEnable & 1<<2))
    {
      /* Disable STD2 by clearing the DAC-enable bit too */
      if (LoaderCode)
      {
        BX_LOADERCMD_MASKWRITEWORD(LoaderCode+*LoaderLength,BX_REG_STD2_CTRL_REG,0,1<<6,*LoaderLength)
      }
      else
      {
        BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD2_CTRL_REG,sizeof(bx_int16),1<<6,0));
      }
    }
    if (!(DecoderEnable & 1<<4))
    {
      /* Disable STD4 by clearing the DAC-enable bit too */
      if (LoaderCode)
      {
        BX_LOADERCMD_MASKWRITEWORD(LoaderCode+*LoaderLength,BX_REG_STD4_CTRL_REG,0,1<<6,*LoaderLength)
      }
      else
      {
        BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_STD4_CTRL_REG,sizeof(bx_int16),1<<6,0));
      }
    }
    /***********************************************************************************/
    /* HW bugfix: END                                                                  */
    /***********************************************************************************/
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTDecoderRead
 *
 * Purpose: Read decoder-props from card to host memory
 *---------------------------------------------------------------------------*/
bx_errtype BestXTDecoderRead (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXTDecoderRead [tdecoderread]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_TRY (TDecoderReadCheck (handle, BX_FALSE, NULL));
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXTDecoderCheck
 *
 * Purpose: Check decoder-props from card to host memory
 *---------------------------------------------------------------------------*/
bx_errtype BestXTDecoderCheck (bx_handletype handle, bx_bool * pResult)
{
  BX_DECLARE_FUNCNAME("BestXTDecoderCheck [tdeccheck]");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    BX_TRY (TDecoderReadCheck (handle, BX_TRUE, pResult));
  }

  BX_ERRETURN (BX_TRY_RET);
}

/*****************************************************************************
 * static functions, as declared on top of file
 *****************************************************************************/
/*---------------------------------------------------------------------------*
 * static bx_errtype TDecoderReadCheck (bx_handletype handle,
 *
 * Purpose      : read decoder properties from card;
 *                store values in host-db (doCheck == BX_FALSE) or
 *                compare against values in host-db (doCheck == BX_TRUE)
 *---------------------------------------------------------------------------*/
static bx_errtype TDecoderReadCheck (bx_handletype handle,
                                     bx_bool doCheck,
                                     bx_bool * pResult)
{
  BX_DECLARE_FUNCNAME("TDecoderReadCheck");
  BX_TRY_VARS_NO_PROG;

  /*
    This reads all decoder properties, 
    WHICH HAVE NO REPRESENTATION IN CONFIG SPACE,
    (i.e. RESOURCE,RESBASE,RESSIZE and SIZE (for config and requester decoder only)
    from card to host. All other decoder properties 
    are implicitely read with the config space 
    (BestXCTConfigRead()).
  */

  /* Decoder properties */
  bx_int32 PropResource; 
  bx_int32 PropResSize;  
  bx_int32 PropResBase; 
  
  /* These values are read for each BAR and the Expansion ROM decoder */
  bx_int16 DecoderControl; /* Type,resource,properties of a decoder         */
  bx_int32 DecoderResSize; /* Size of the internal resource                 */
  bx_int32 DecoderResBase; /* Offset for the internal resource              */

  /* Read from card for config and requester decoder */
  bx_int32 DecoderEnable;
  
  bx_int32 Bar;          
  
  BX_TRY_BEGIN 
  {
    if (doCheck)
    {
      BX_TRY_FCT_PARAM_NULL_POINTER (pResult);
      *pResult = BX_TRUE;   /* will be set to false in case a comparison fails */
    }

    /**************************************************************************
     * STD/BAR decoders
     **************************************************************************/

    /* Handle all pairs of BARs */
    for (Bar=0;Bar<6;Bar++)
    {
      /* Read control, resbase and ressize registers from HW */
      BX_TRY(BestXDirectRegRead(handle,DecAddr[Bar].Control,
                                sizeof(bx_int16),(bx_int32*)&DecoderControl));
      BX_TRY(BestXDirectRegRead(handle,DecAddr[Bar].ResBase,
                                sizeof(bx_int32),&DecoderResBase));
      BX_TRY(BestXDirectRegRead(handle,DecAddr[Bar].ResSize,
                                sizeof(bx_int32),&DecoderResSize));

      /* BX_DECP_RESOURCE: Bits 0-2 of Control register */
      switch (DecoderControl&7)
      {
       case 0:  /* data memory */ 
        PropResource=BX_DECP_RESOURCE_MEM;
        break;          
       case 1:  /* data generator */ 
        PropResource=BX_DECP_RESOURCE_GEN;
        break;          
       case 2:  /* config space */ 
        /* 19.8.2003 added for SR
            Caution: BX_DECP_RESOURCE_CFG works only with IO decoder.
            Problems with MEM decoders (WAITS)
        */
        PropResource=BX_DECP_RESOURCE_CFG;
        break;           
#if 0
      case 3:  /* DBI */ 
      PropResource=BX_DECP_RESOURCE_FLASH;
      break;           
#endif
       default:
        BX_TRY_FAIL(BX_E_INVALID_CASE); /* Invalid value in HW register (bit 2 has to be always 0) */
        break;
      }
      
      /* BX_DECP_RESBASE */
      PropResBase=DecoderResBase;

      /* BX_DECP_RESSIZE
         Bit 0,1,24,..,31 are always fix zero in HW, so PropResSize in {2,..,24} (24=all zeros),
         i.e. size in bytes is always DW aligned and zero not allowed !!
      */
       
      PropResSize=Log2( ~(DecoderResSize | (0xFFFFFFFF << 24))+1 );
#if 0
      for (i=2;i<=23;i++) /* bits 2-23 are relevant, rest is fix zero in HW */
      {
        /* Look for the lowest bit set 
           (format is someting like FFFFFF00 for 256 bytes, i.e. size=8) */
        if ((1<<i) & DecoderResSize)
        {
          PropResSize=i; /* i.e. 1<<i bytes */
        }
      }
#endif
      if (PropResSize==0)
      {
        /* no set bit found in previous loop */
        assert(DecoderResSize==0x0); /* bits 0,1,24,..,31 are fix zero in HW !! */
        PropResSize=24; /* This is the maximum possible size */
      }
      

      if (doCheck)
      {
        bx_int32 cmpVal;
        
        /* compare agains properties on host */
        BX_TRY(BestXTDecoderGet(handle, Bar, BX_DECP_RESOURCE, &cmpVal));
        BX_TRY_FAIL (cmpVal != PropResource ? BX_E_INTERNAL_RETURN : BX_E_OK);

        BX_TRY(BestXTDecoderGet(handle, Bar, BX_DECP_RESBASE,  &cmpVal));
        BX_TRY_FAIL (cmpVal != PropResBase ? BX_E_INTERNAL_RETURN : BX_E_OK);

        BX_TRY(BestXTDecoderGet(handle, Bar, BX_DECP_RESSIZE,  &cmpVal));
        BX_TRY_FAIL (cmpVal != PropResSize ? BX_E_INTERNAL_RETURN : BX_E_OK);
      }
      else
      {
        /* Set properties on host for BAR Bar */
        BX_TRY(BestXTDecoderSet(handle,Bar,BX_DECP_RESOURCE, PropResource));
        BX_TRY(BestXTDecoderSet(handle,Bar,BX_DECP_RESBASE,  PropResBase));
        BX_TRY(BestXTDecoderSet(handle,Bar,BX_DECP_RESSIZE,  PropResSize));
      }

    } /* for bar... */
      
    /**************************************************************************
     * Expansion ROM decoder
     **************************************************************************/

    /* Read control, resbase and ressize registers from HW */
    BX_TRY(BestXDirectRegRead(handle,DecAddr[BX_DEC_EXPROM].Control,sizeof(bx_int16),(bx_int32 *)&DecoderControl));
    BX_TRY(BestXDirectRegRead(handle,DecAddr[BX_DEC_EXPROM].ResBase,sizeof(bx_int32),&DecoderResBase));
    BX_TRY(BestXDirectRegRead(handle,DecAddr[BX_DEC_EXPROM].ResSize,sizeof(bx_int32),&DecoderResSize));

    /* BX_DECP_RESOURCE: Bits 0-2 of Control register */
    switch (DecoderControl&7)
    {
     case 0:  /* data memory */ 
      PropResource=BX_DECP_RESOURCE_MEM;
      break;          
     case 1:  /* data generator */ 
      PropResource=BX_DECP_RESOURCE_GEN;
      break;          
#if 0
      case 2:  /* config space */ 
      PropResource=BX_DECP_RESOURCE_CFG;
      break;
#endif
     case 3:  /* DBI */ 
      PropResource=BX_DECP_RESOURCE_FLASH;
      break;          
     default:
      PropResource=BX_DECP_RESOURCE_MEM;
      BX_TRY_FAIL(BX_E_INVALID_CASE); /* Invalid value in HW register (bit 2 has to be always 0) */
      break;
    }
      
    /* BX_DECP_RESBASE */
    if (PropResource == BX_DECP_RESOURCE_FLASH)
    {  /* if resource is FLASH set base and size to fix values */
       PropResBase = BX_EXPROM_IMAGE;
    }
    else
    {
       PropResBase=DecoderResBase;
    }
    /* BX_DECP_RESSIZE
       Bit 0,1,24,..,31 are always fix zero in HW, so PropResSize in {2,..,24} (24=all zeros),
       i.e. size in bytes is always DW aligned and zero not allowed !!
    */
       

    if (PropResource == BX_DECP_RESOURCE_FLASH)
    {  /* if resource is FLASH set base and size to fix values */
       PropResSize=Log2(BX_EXPROM_IMAGE_SIZE);
    }
    else
    {
       PropResSize=Log2( ~(DecoderResSize | (0xFFFFFFFF << 24))+1 );
    }

#if 0
    PropResSize=0; /* invalid !! */
    for (i=2;i<=23;i++) /* bits 2-23 are relevant, rest is fix zero in HW */
    {
      /* Look for the lowest bit set 
         (format is someting like FFFFFF00 for 256 bytes, i.e. size=8) */
      if ((1<<i) & DecoderResSize)
      {
        PropResSize=i; /* i.e. 1<<i bytes */
      }
    }
#endif
    if (PropResSize==0)
    {
      /* no set bit found in previous loop */
      assert(DecoderResSize==0x0); /* bits 0,1,24,..,31 are fix zero in HW !! */
      PropResSize=24; /* This is the maximum possible size */
    }

    if (doCheck)
    {
      bx_int32 cmpVal;
        
      /* compare agains properties on host */
      BX_TRY(BestXTDecoderGet(handle, BX_DEC_EXPROM, BX_DECP_RESOURCE, &cmpVal));
      BX_TRY_FAIL (cmpVal != PropResource ? BX_E_INTERNAL_RETURN : BX_E_OK);

      BX_TRY(BestXTDecoderGet(handle, BX_DEC_EXPROM, BX_DECP_RESBASE,  &cmpVal));
      BX_TRY_FAIL (cmpVal != PropResBase ? BX_E_INTERNAL_RETURN : BX_E_OK);

      BX_TRY(BestXTDecoderGet(handle, BX_DEC_EXPROM, BX_DECP_RESSIZE,  &cmpVal));
      BX_TRY_FAIL (cmpVal != PropResSize ? BX_E_INTERNAL_RETURN : BX_E_OK);
    }
    else
    {
      /* Set properties on host for BAR Bar */
      BX_TRY(BestXTDecoderSet(handle,BX_DEC_EXPROM,BX_DECP_RESOURCE, PropResource));
      BX_TRY(BestXTDecoderSet(handle,BX_DEC_EXPROM,BX_DECP_RESBASE,  PropResBase));
      BX_TRY(BestXTDecoderSet(handle,BX_DEC_EXPROM,BX_DECP_RESSIZE,  PropResSize));
    }
      
    /**************************************************************************
     * Config decoder
     **************************************************************************/
    /**************************************************************************
     * Requester target decoder
     **************************************************************************/
     
    /* These two decoders only support enabling/disabling.
       That information is represented in BX_DECP_SIZE, 
       which is usually (for all other decoders) coded implicitely 
       in config space (see BestXCTConfigRead())
    */

    /* Read the decoder enable register */  
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_EN_SREG,sizeof(bx_int16),&DecoderEnable));

    if (doCheck)
    {
      bx_int32 cmpVal;
      /* compare the decoder properties */
      BX_TRY(BestXTDecoderGet(handle,BX_DEC_CONFIG,BX_DECP_SIZE, &cmpVal));
      BX_TRY_FAIL ( (cmpVal != (DecoderEnable & 1UL << 8UL ? 1UL : 0UL)) ?
                    BX_E_INTERNAL_RETURN : BX_E_OK );

      BX_TRY(BestXTDecoderGet(handle,BX_DEC_RT,BX_DECP_SIZE, &cmpVal));
      BX_TRY_FAIL ( (cmpVal != (DecoderEnable & 1UL << 7UL ? 1UL : 0UL )) ?
                    BX_E_INTERNAL_RETURN : BX_E_OK );
    }
    else
    {
      /* Set the decoder properties */
      BX_TRY(BestXTDecoderSet(handle,BX_DEC_CONFIG,BX_DECP_SIZE,(DecoderEnable&1<<8?1:0)));
      BX_TRY(BestXTDecoderSet(handle,BX_DEC_RT,BX_DECP_SIZE,(DecoderEnable&1<<7?1:0)));
    }
  }

  BX_TRY_CATCH {
    if (BX_TRY_RET == BX_E_INTERNAL_RETURN) /* we get here if some comparison has failed! */
    {
      *pResult = BX_FALSE;
      BX_TRY_RET = BX_E_OK;
    }
  }
  
  BX_ERRETURN(BX_TRY_RET);
}
    
/*---------------------------------------------------------------------------*
 * static bx_errtype ConfigValueRead
 *
 * Purpose      : read standard config space into value field
 *---------------------------------------------------------------------------*/
static bx_errtype ConfigValueRead (bx_handletype handle, bx_int32ptr pValue)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    /* Read standard config space (see xsetting.c):
       24 DWs=0x60 Bytes: 00-3c and 0x60-0x7c  for Mephisto (PCI address view)
       28 DWs=0x70 Bytes: 00-3c and 0x60-0x8c  for Faust    (PCI address view)
       Caution:
       Range 40-5c (8 programming registers) are only visible from PCI,
       so this data is/cannot be read here !
    */
    BX_TRY(BestXDirectRegBlockRead( handle,
                                    BX_REG_DEV_VEN_ID_REG, /* address */
                                    4UL,                   /* regsize */
                                    0UL,                   /* no autoinc */
                                    (bx_int8ptr) pValue,   /* data Ptr */
                                    4UL,                   /* DWORD data */
                                    (BestXHasMephisto(handle)?0x60:0x70))); /* size of data */

    /* Copy extended configspace into right pos. of cfgspace array:
       This moves the second part of the data down (to correct PCI address) 
       in order to get room for the left out programming registers.
       Caution: In case of Faust source and destination are overlapping !! */
    memmove (pValue+0x18,pValue+0x10,(BestXHasMephisto(handle)?0x8:0xc)*sizeof(bx_int32));
    
    /* Set invalid stuff (represents 8 PCI programming registers) to zero */
    memset (pValue+0x10, 0, 8 * sizeof (bx_int32) );
  }

  return (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype ConfigMaskRead
 *
 * Purpose      : read standard config space masks into value field
 *---------------------------------------------------------------------------*/
static bx_errtype ConfigMaskRead (bx_handletype handle, bx_int32ptr pMask)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    /* Read standard config space (see xsetting.c):
       24 DWs=0x60 Bytes: 00-3c and 0x60-0x7c  for Mephisto (PCI address view)
       28 DWs=0x70 Bytes: 00-3c and 0x60-0x8c  for Faust    (PCI address view)
       Caution:
       Range 40-5c (8 programming registers) are only visible from PCI,
       so this data is/cannot be read here !
    */
    BX_TRY(BestXDirectRegBlockRead( handle,
                                    BX_REG_DEV_VEN_ID_MASK_REG, /* address */
                                    4UL,                   /* regsize */
                                    0UL,                   /* no autoinc */
                                    (bx_int8ptr) pMask,    /* data Ptr */
                                    4UL,                   /* DWORD data */
                                    (BestXHasMephisto(handle)?0x60:0x70))); /* size of data */

    /* Copy extended configspace into right pos. of cfgspace array:
       This moves the second part of the data down (to correct PCI address) 
       in order to get room for the left out programming registers.
       Caution: In case of Faust source and destination are overlapping !! */
    memmove (pMask+0x18,pMask+0x10,(BestXHasMephisto(handle)?0x8:0xc)*sizeof(bx_int32));
    
    /* Set invalid stuff (represents 8 PCI programming registers) to zero */
    memset (pMask+0x10, 0, 8 * sizeof (bx_int32) );
  }

  return (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype ConfigValueProg
 *
 * Purpose      : prog standard config space into value field
 *---------------------------------------------------------------------------*/
static bx_errtype ConfigValueProg (bx_handletype handle, bx_int32ptr pValue)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    /* copy extended configspace into right pos. of cfgspace array */
    memmove (pValue+0x10,pValue+0x18,(BestXHasMephisto(handle)?0x8:0xc)*sizeof(bx_int32));
    
    /* prog standard config space */
    BX_TRY(BestXDirectRegBlockWrite( handle,
                                     BX_REG_DEV_VEN_ID_REG, /* address */
                                     4UL,                  /* regsize */
                                     0UL,                  /* no autoinc */
                                     (bx_int8ptr) pValue,  /* data Ptr */
                                     4UL,                  /* DWORD data */
                                     (BestXHasMephisto(handle)?0x60:0x70))); /* size of data */

    /* set invalid stuff to zero */
    memset (pValue+0x10, 0, 8 * sizeof (bx_int32) );
  }

  return (BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static bx_errtype ConfigMaskProg
 *
 * Purpose      : prog standard config space masks into value field
 *---------------------------------------------------------------------------*/
static bx_errtype ConfigMaskProg (bx_handletype handle, bx_int32ptr pMask)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN 
  {
    /* copy extended configspace into right pos. of cfgspace array */
    memmove (pMask+0x10,pMask+0x18,(BestXHasMephisto(handle)?0x8:0xc)*sizeof(bx_int32));

    /* prog standard config space */
    BX_TRY(BestXDirectRegBlockWrite( handle,
                                     BX_REG_DEV_VEN_ID_MASK_REG, /* address */
                                     4UL,                  /* regsize */
                                     0UL,                  /* no autoinc */
                                     (bx_int8ptr) pMask,   /* data Ptr */
                                     4UL,                  /* DWORD data */
                                     (BestXHasMephisto(handle)?0x60:0x70))); /* size of data */

    /* set invalid stuff to zero */
    memset (pMask+0x10, 0, 8 * sizeof(bx_int32) );
  }

  return (BX_TRY_RET);
}

